// Final working popup for job application autofill
console.log('Job Autofill popup loaded');

// Profile fields
const fields = [
  'firstName', 'lastName', 'email', 'phone', 'city', 
  'region', 'country', 'linkedin', 'github', 'portfolio', 'summary', 'interests',
  'state', 'salary', 'startDate', 'education', 'school', 'workAuth', 'location', 'nycOffice', 'address', 'experience',
  'organizedExample', 'lookerExperience', 'spreadsheetExample', 'visaSponsorship',
  // gender fields
  'gender', 'genderSelfDescribe'
];

// Update status message
function showStatus(message, isError = false) {
  const statusEl = document.getElementById('status');
  statusEl.textContent = message;
  statusEl.style.color = isError ? 'red' : '#333';
  statusEl.style.background = isError ? '#ffe6e6' : '#f0f0f0';
  console.log('Status:', message);
}

// Load saved profile from storage
function loadProfile() {
  try {
    chrome.storage.local.get(['jobProfile'], function(result) {
      if (result.jobProfile) {
        const profile = result.jobProfile;
        fields.forEach(field => {
          const element = document.getElementById(field);
          if (element) {
            element.value = profile[field] || '';
          }
        });
        
        // Load resume file info
        if (profile.resumeFileName) {
          console.log('📄 Resume data found in profile:', {
            fileName: profile.resumeFileName,
            fileSize: profile.resumeFileSize,
            hasFileData: !!profile.resumeFileData,
            fileDataLength: profile.resumeFileData ? profile.resumeFileData.length : 0
          });
          
          const resumeStatus = document.getElementById('resumeStatus');
          if (resumeStatus) {
            resumeStatus.textContent = `📄 Saved: ${profile.resumeFileName}`;
            resumeStatus.style.color = '#4CAF50';
          }
          
          // Update the file input display to show the saved file name
          const fileInput = document.getElementById('resume');
          if (fileInput && fileInput.nextElementSibling) {
            const noFileText = fileInput.nextElementSibling;
            if (noFileText && (noFileText.textContent === 'No file chosen' || noFileText.textContent.trim() === '')) {
              noFileText.textContent = profile.resumeFileName;
              noFileText.style.color = '#4CAF50';
              noFileText.style.fontWeight = 'bold';
            }
          }
        } else {
          console.log('ℹ️ No resume data found in profile');
        }
        
        showStatus('✅ Profile loaded successfully!');
        console.log('Profile loaded:', profile);
      } else {
        showStatus('📝 No saved profile found. Fill in your information below.');
      }
    });
  } catch (error) {
    showStatus('❌ Error loading profile: ' + error.message, true);
    console.error('Load error:', error);
  }
}

// Save profile to storage
function saveProfile() {
  showStatus('💾 Saving your profile...');
  
  try {
    // First get the existing profile to preserve resume data
    chrome.storage.local.get(['jobProfile'], function(result) {
      const existingProfile = result.jobProfile || {};
      
      // Create new profile with form fields
      const profile = {};
      fields.forEach(field => {
        const element = document.getElementById(field);
        if (element) {
          profile[field] = element.value.trim();
        }
      });
      
      // Preserve resume data from existing profile
      if (existingProfile.resumeFileName) {
        profile.resumeFileName = existingProfile.resumeFileName;
        profile.resumeFileSize = existingProfile.resumeFileSize;
        profile.resumeFileType = existingProfile.resumeFileType;
        profile.resumeLastModified = existingProfile.resumeLastModified;
        profile.resumeFileData = existingProfile.resumeFileData;
        console.log('✅ Preserved resume data:', {
          fileName: existingProfile.resumeFileName,
          fileSize: existingProfile.resumeFileSize,
          hasFileData: !!existingProfile.resumeFileData
        });
      } else {
        console.log('ℹ️ No existing resume data to preserve');
      }
      
      console.log('Saving profile with preserved resume data:', profile);
      
      chrome.storage.local.set({jobProfile: profile}, function() {
        if (chrome.runtime.lastError) {
          showStatus('❌ Error saving: ' + chrome.runtime.lastError.message, true);
        } else {
          showStatus('✅ Profile saved successfully! Your information is now stored.');
          console.log('Profile saved successfully');
        }
      });
    });
    
  } catch (error) {
    showStatus('❌ Error saving: ' + error.message, true);
    console.error('Save error:', error);
  }
}

// Fill the current page with saved profile
function fillPage() {
  showStatus('🚀 Filling the page with your information...');
  
  try {
    chrome.tabs.query({active: true, currentWindow: true}, function(tabs) {
      if (tabs[0]) {
        chrome.scripting.executeScript({
          target: {tabId: tabs[0].id},
          func: function() {
            console.log('Dispatching fill event to content script');
            window.dispatchEvent(new CustomEvent('JOB_AUTOFILL_NOW'));
          }
        }, function() {
          if (chrome.runtime.lastError) {
            showStatus('❌ Error filling page: ' + chrome.runtime.lastError.message, true);
          } else {
            showStatus('✅ Fill command sent! Check the page for filled fields.');
          }
        });
      }
    });
    
  } catch (error) {
    showStatus('❌ Error filling page: ' + error.message, true);
    console.error('Fill error:', error);
  }
}

// Handle resume file upload
function setupResumeUpload() {
  console.log('Setting up resume upload...');
  
  const resumeInput = document.getElementById('resume');
  console.log('Resume input element:', resumeInput);
  
  if (resumeInput) {
    console.log('Resume input found, adding event listener');
    
    resumeInput.addEventListener('change', async (e) => {
      console.log('Resume file changed!');
      const file = e.target.files[0];
      console.log('Selected file:', file);
      
      if (file) {
        console.log('File details:', {
          name: file.name,
          size: file.size,
          type: file.type
        });
        
        try {
          showStatus('💾 Saving resume...');
          
          // Convert file to base64 for storage
          const fileReader = new FileReader();
          fileReader.onload = async function(e) {
            try {
              const base64Data = e.target.result;
              console.log('File converted to base64, length:', base64Data.length);
              
              // Store file info and data for easy access
              const { jobProfile } = await chrome.storage.local.get(['jobProfile']);
              console.log('Current profile:', jobProfile);
              
              const updatedProfile = { 
                ...jobProfile, 
                resumeFileName: file.name,
                resumeFileSize: file.size,
                resumeFileType: file.type,
                resumeLastModified: file.lastModified,
                resumeFileData: base64Data
              };
              
              console.log('Updated profile:', updatedProfile);
              
              await chrome.storage.local.set({ jobProfile: updatedProfile });
              console.log('Profile saved to storage with file data');
              
              const resumeStatus = document.getElementById('resumeStatus');
              console.log('Resume status element:', resumeStatus);
              
              if (resumeStatus) {
                resumeStatus.textContent = `📄 Saved: ${file.name} (${(file.size/1024/1024).toFixed(2)} MB)`;
                resumeStatus.style.color = '#4CAF50';
                console.log('Status updated');
              }
              
              // Update the file input to show the selected file
              const fileInput = document.getElementById('resume');
              if (fileInput && fileInput.nextElementSibling) {
                const noFileText = fileInput.nextElementSibling;
                if (noFileText && noFileText.textContent === 'No file chosen') {
                  noFileText.textContent = file.name;
                  noFileText.style.color = '#4CAF50';
                  noFileText.style.fontWeight = 'bold';
                }
              }
              
              showStatus(`✅ Resume saved: ${file.name}`);
              console.log('Resume saved successfully');
              
            } catch (error) {
              console.error('Resume save error:', error);
              showStatus('❌ Error saving resume');
            }
          };
          
          fileReader.onerror = function() {
            console.error('Error reading file');
            showStatus('❌ Error reading resume file');
          };
          
          fileReader.readAsDataURL(file);
          
        } catch (error) {
          console.error('Resume save error:', error);
          showStatus('❌ Resume save failed: ' + error.message, true);
          
          const resumeStatus = document.getElementById('resumeStatus');
          if (resumeStatus) {
            resumeStatus.textContent = '❌ Save failed';
            resumeStatus.style.color = '#f44336';
          }
        }
      } else {
        console.log('No file selected');
        showStatus('No file selected', true);
      }
    });
  } else {
    console.error('Resume input element not found!');
    showStatus('❌ Resume input not found', true);
  }
}

// Set up event listeners when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
  console.log('DOM loaded, setting up job autofill popup');
  
  // Load existing profile
  loadProfile();
  
  // Setup resume upload
  setupResumeUpload();
  
  // Debug: Check if resume elements exist
  setTimeout(() => {
    const resumeInput = document.getElementById('resume');
    const resumeStatus = document.getElementById('resumeStatus');
    console.log('DEBUG - Resume input exists:', !!resumeInput);
    console.log('DEBUG - Resume status exists:', !!resumeStatus);
    console.log('DEBUG - All elements with "resume" in ID:', document.querySelectorAll('[id*="resume"]'));
    console.log('DEBUG - All file inputs:', document.querySelectorAll('input[type="file"]'));
    console.log('DEBUG - Document body HTML length:', document.body.innerHTML.length);
    console.log('DEBUG - Looking for "Resume File" text:', document.body.innerHTML.includes('Resume File'));
  }, 100);
  
  // Save button
  document.getElementById('saveBtn').addEventListener('click', function() {
    console.log('Save button clicked');
    saveProfile();
  });
  
  // Fill button
  document.getElementById('fillBtn').addEventListener('click', function() {
    console.log('Fill button clicked');
    fillPage();
  });
  
  console.log('Event listeners set up successfully');
  
  // Emergency: Add resume field if it doesn't exist
  setTimeout(() => {
    const resumeInput = document.getElementById('resume');
    if (!resumeInput) {
      console.log('Resume field missing - adding it manually');
      
      // Find the summary field and add resume field after it
      const summaryField = document.getElementById('summary');
      if (summaryField) {
        const parent = summaryField.parentElement.parentElement;
        const resumeDiv = document.createElement('div');
        resumeDiv.className = 'field';
        resumeDiv.innerHTML = `
          <label>Resume File</label>
          <input type="file" id="resume" accept=".pdf,.doc,.docx,.txt" style="width: 100%; padding: 6px; border: 1px solid #ccc; border-radius: 3px;">
          <small style="color: #666; font-size: 11px;">Upload your resume (PDF, DOC, DOCX, TXT)</small>
          <div id="resumeStatus" style="font-size: 12px; margin-top: 4px;"></div>
        `;
        
        parent.insertBefore(resumeDiv, parent.children[parent.children.length - 3]); // Insert before "Additional Job Application Info"
        
        // Re-setup resume upload with the new field
        setupResumeUpload();
        console.log('Resume field added manually');
      }
    }
  }, 200);
});

// Manual test function for resume upload
window.testResumeUpload = function() {
  console.log('=== MANUAL RESUME UPLOAD TEST ===');
  
  const resumeInput = document.getElementById('resume');
  const resumeStatus = document.getElementById('resumeStatus');
  
  console.log('Resume input element:', resumeInput);
  console.log('Resume status element:', resumeStatus);
  
  if (resumeInput) {
    console.log('✅ Resume input found!');
    console.log('Input type:', resumeInput.type);
    console.log('Input accept:', resumeInput.accept);
    
    // Simulate a file selection
    console.log('You can now click the resume input to test file selection');
  } else {
    console.log('❌ Resume input NOT found!');
    console.log('Available elements with "resume":', document.querySelectorAll('[id*="resume"]'));
    console.log('All file inputs:', document.querySelectorAll('input[type="file"]'));
  }
  
  if (resumeStatus) {
    console.log('✅ Resume status element found!');
  } else {
    console.log('❌ Resume status element NOT found!');
  }
};
