// Ultra aggressive content script - WILL fill phone fields
// Version: 2.1 - Added Chrome API error handling
console.log('🚀 ULTRA AGGRESSIVE CONTENT SCRIPT LOADED - VERSION 2.1 🚀');
console.log('✅ This version includes Chrome API error handling');
console.log('✅ If you see this message, the new version is active');
console.log('⏰ Loaded at:', new Date().toLocaleTimeString());

// Function to format phone number for better compatibility
function formatPhoneNumber(phone) {
  if (!phone) return '';
  
  // Remove all non-digit characters
  const digits = phone.replace(/\D/g, '');
  
  // Format based on length - always add +1 for US numbers
  if (digits.length === 10) {
    return `+1${digits}`;
  } else if (digits.length === 11 && digits[0] === '1') {
    return `+${digits}`;
  } else if (digits.length === 11) {
    return `+${digits}`;
  }
  
  // Return original if can't format
  return phone;
}

// Function to fill ANY field that could be phone
function fillPhoneField(value) {
  if (!value || value.trim() === '') return false;
  
  console.log(`Looking for phone field with value: ${value}`);
  
  // Get ALL input fields on the page
  const allInputs = document.querySelectorAll('input, textarea, select');
  console.log(`Found ${allInputs.length} total form elements`);
  
  let filled = false;
  
  for (const input of allInputs) {
    try {
      const name = (input.name || '').toLowerCase();
      const id = (input.id || '').toLowerCase();
      const placeholder = (input.placeholder || '').toLowerCase();
      const type = (input.type || '').toLowerCase();
      const className = (input.className || '').toLowerCase();
      
      // Get all text around this input
      let parentText = '';
      let labelText = '';
      let siblingText = '';
      
      if (input.parentElement) {
        parentText = input.parentElement.textContent.toLowerCase();
      }
      
      // Look for associated label
      const label = input.closest('label') || 
                   document.querySelector(`label[for="${input.id}"]`) ||
                   input.parentElement?.querySelector('label');
      if (label) {
        labelText = label.textContent.toLowerCase();
      }
      
      // Look at sibling elements
      if (input.previousElementSibling) {
        siblingText = input.previousElementSibling.textContent.toLowerCase();
      }
      
      console.log(`Checking input: name="${input.name}", id="${input.id}", type="${input.type}"`);
      console.log(`Parent text: "${parentText.substring(0, 100)}..."`);
      console.log(`Label text: "${labelText}"`);
      console.log(`Sibling text: "${siblingText}"`);
      
      // SUPER AGGRESSIVE phone detection
      let isPhoneField = false;
      
      // Check if it's explicitly a phone field
      if (type === 'tel' || 
          name.includes('phone') || name.includes('tel') || name.includes('mobile') || name.includes('cell') ||
          id.includes('phone') || id.includes('tel') || id.includes('mobile') || id.includes('cell') ||
          placeholder.includes('phone') || placeholder.includes('tel') || placeholder.includes('mobile') ||
          labelText.includes('phone') || labelText.includes('tel') || labelText.includes('mobile') ||
          parentText.includes('phone') || parentText.includes('tel') || parentText.includes('mobile') ||
          siblingText.includes('phone') || siblingText.includes('tel') || siblingText.includes('mobile')) {
        isPhoneField = true;
        console.log('✅ Found explicit phone field');
      }
      
      // Check if it's next to a country field (common pattern)
      if (parentText.includes('country') && (parentText.includes('phone') || parentText.includes('tel'))) {
        isPhoneField = true;
        console.log('✅ Found phone field next to country');
      }
      
      // Check if it's a text input that could be phone based on context
      if (input.tagName === 'INPUT' && (type === 'text' || type === 'tel' || !type)) {
        // Look for phone-like patterns in surrounding text
        if (parentText.includes('contact') || parentText.includes('number') || 
            parentText.includes('call') || parentText.includes('reach') ||
            labelText.includes('contact') || labelText.includes('number') ||
            siblingText.includes('contact') || siblingText.includes('number')) {
          isPhoneField = true;
          console.log('✅ Found potential phone field by context');
        }
      }
      
      // Check if it's positioned like a phone field (next to country, after email, etc.)
      const rect = input.getBoundingClientRect();
      if (rect.width > 80 && rect.height > 20 && rect.height < 50) { // Reasonable size for phone
        // Look for country field nearby
        const countryFields = document.querySelectorAll('input, select');
        for (const field of countryFields) {
          const fieldRect = field.getBoundingClientRect();
          const fieldText = field.parentElement ? field.parentElement.textContent.toLowerCase() : '';
          if (fieldText.includes('country') && 
              Math.abs(rect.top - fieldRect.top) < 100 && 
              Math.abs(rect.left - fieldRect.left) < 500) {
            isPhoneField = true;
            console.log('✅ Found phone field by position near country field');
            break;
          }
        }
      }
      
      if (isPhoneField) {
        console.log(`🎯 FOUND PHONE FIELD:`, input);
        
        // Fill the field with better event handling
        input.focus();
        
        // Clear any existing value first
        input.value = '';
        
        // Set the value
        input.value = value;
        
        // Ensure text is visible
        ensureTextVisibility(input);
        
        // Trigger events in the correct order
        input.dispatchEvent(new Event('focus', { bubbles: true }));
        input.dispatchEvent(new Event('input', { bubbles: true }));
        input.dispatchEvent(new Event('change', { bubbles: true }));
        input.dispatchEvent(new Event('blur', { bubbles: true }));
        
        // Force the value to stick
        setTimeout(() => {
          if (input.value !== value) {
            input.value = value;
            ensureTextVisibility(input);
            input.dispatchEvent(new Event('input', { bubbles: true }));
          }
        }, 100);
        
        // Highlight the filled field
        input.style.outline = '3px solid #4CAF50';
        input.style.backgroundColor = '#e8f5e8';
        
        console.log(`✅ FILLED PHONE FIELD: ${value}`);
        filled = true;
        break; // Stop after filling the first phone field
      }
    } catch (error) {
      console.log(`Error processing input:`, error);
    }
  }
  
  return filled;
}

// Helper functions
function getLabelText(element) {
  // Check for label with 'for' attribute
  if (element.id) {
    const label = document.querySelector(`label[for="${element.id}"]`);
    if (label) return label.textContent.trim();
  }
  
  // Check for label wrapping the element
  const label = element.closest('label');
  if (label) return label.textContent.trim();
  
  // Check for nearby label elements
  const parent = element.parentElement;
  if (parent) {
    const label = parent.querySelector('label');
    if (label) return label.textContent.trim();
  }
  
  return '';
}

// Ensure filled text is visible
function ensureTextVisibility(element) {
  // Force visible text styling
  element.style.color = '#000000 !important';
  element.style.backgroundColor = '#ffffff !important';
  element.style.opacity = '1 !important';
  
  // Also set important styles to override any CSS
  element.setAttribute('style', element.getAttribute('style') + '; color: #000000 !important; background-color: #ffffff !important; opacity: 1 !important;');
  
  // For dark mode compatibility, also try dark text on light background
  const computedStyle = window.getComputedStyle(element);
  const currentColor = computedStyle.color;
  const currentBg = computedStyle.backgroundColor;
  
  // If the computed color is very light or transparent, force dark text
  if (currentColor === 'rgba(0, 0, 0, 0)' || currentColor === 'transparent' || 
      currentColor.includes('255, 255, 255') || currentColor.includes('rgba(255, 255, 255')) {
    element.style.setProperty('color', '#000000', 'important');
  }
  
  // If background is very dark, ensure contrast
  if (currentBg && (currentBg.includes('0, 0, 0') || currentBg.includes('rgba(0, 0, 0'))) {
    element.style.setProperty('background-color', '#ffffff', 'important');
    element.style.setProperty('color', '#000000', 'important');
  }
}

// Use the native value setter so React-style controlled inputs detect the change
function setNativeInputValue(input, value) {
  try {
    const proto = Object.getPrototypeOf(input) || HTMLInputElement.prototype;
    const valueSetter = Object.getOwnPropertyDescriptor(proto, 'value')?.set ||
                        Object.getOwnPropertyDescriptor(HTMLInputElement.prototype, 'value')?.set;
    if (valueSetter) {
      valueSetter.call(input, value);
    } else {
      input.value = value;
    }
  } catch (_) {
    input.value = value;
  }
}

// Snapshot and restore text-like inputs to avoid re-render clearing values when radios are toggled
function snapshotTextInputs() {
  try {
    const selectors = [
      'input[type="text"]', 'input[type="email"]', 'input[type="url"]', 'input:not([type])', 'textarea'
    ];
    const elements = Array.from(document.querySelectorAll(selectors.join(',')));
    return elements.map(el => ({ el, value: el.value }));
  } catch (_) { return []; }
}

function restoreTextInputs(snapshot) {
  try {
    snapshot.forEach(({ el, value }) => {
      if (!el || !el.isConnected) return;
      if (el.value !== value) {
        setNativeInputValue(el, value);
        el.dispatchEvent(new Event('input', { bubbles: true }));
        el.dispatchEvent(new Event('change', { bubbles: true }));
      }
    });
  } catch (_) {}
}

// Simulate real typing so React/Ashby controlled inputs accept the value
function typeLikeHuman(input, value) {
  try {
    input.focus();
    setNativeInputValue(input, '');
    input.dispatchEvent(new Event('input', { bubbles: true }));
    for (const ch of String(value)) {
      input.setSelectionRange?.(input.value.length, input.value.length);
      input.dispatchEvent(new KeyboardEvent('keydown', { bubbles: true, key: ch, code: undefined }));
      input.dispatchEvent(new KeyboardEvent('keypress', { bubbles: true, key: ch, code: undefined }));
      const next = (input.value || '') + ch;
      setNativeInputValue(input, next);
      input.dispatchEvent(new InputEvent('input', { bubbles: true, inputType: 'insertText', data: ch }));
      input.dispatchEvent(new KeyboardEvent('keyup', { bubbles: true, key: ch, code: undefined }));
    }
    input.dispatchEvent(new Event('change', { bubbles: true }));
    input.dispatchEvent(new Event('blur', { bubbles: true }));
    // Verify and enforce
    setTimeout(() => {
      if (input.value !== value) {
        setNativeInputValue(input, value);
        input.dispatchEvent(new Event('input', { bubbles: true }));
        input.dispatchEvent(new Event('change', { bubbles: true }));
      }
    }, 80);
  } catch (_) {
    setNativeInputValue(input, value);
    input.dispatchEvent(new Event('input', { bubbles: true }));
    input.dispatchEvent(new Event('change', { bubbles: true }));
  }
}

function getParentText(element) {
  const parent = element.closest('div, fieldset, form, section, p');
  return parent ? parent.textContent.trim() : '';
}

// Enhanced LinkedIn field detection
function isLinkedInField(element) {
  const name = (element.name || '').toLowerCase();
  const id = (element.id || '').toLowerCase();
  const placeholder = (element.placeholder || '').toLowerCase();
  const type = (element.type || '').toLowerCase();
  const className = (element.className || '').toLowerCase();
  
  const labelText = getLabelText(element).toLowerCase();
  const parentText = getParentText(element).toLowerCase();
  
  // LinkedIn-specific patterns
  const linkedinPatterns = [
    'linkedin', 'linked_in', 'linked-in', 'li_', 'linkedin_',
    'social', 'social_media', 'social_profile', 'social_url',
    'profile', 'profile_url', 'professional_profile'
  ];
  
  // Special LinkedIn indicators
  const isLinkedInPlaceholder = placeholder === 'https://' || placeholder.startsWith('https://');
  const isLinkedInUrlField = type === 'url' && (isLinkedInPlaceholder || !name && !id && !labelText);
  
  if (isLinkedInPlaceholder) {
    console.log(`🎯 LINKEDIN PLACEHOLDER DETECTED: "${placeholder}"`, element);
  }
  
  // Check all attributes for LinkedIn patterns
  const allText = `${name} ${id} ${placeholder} ${className} ${labelText} ${parentText}`;
  
  for (const pattern of linkedinPatterns) {
    if (allText.includes(pattern)) {
      console.log(`🎯 LINKEDIN FIELD DETECTED by pattern "${pattern}":`, element);
      return true;
    }
  }
  
  // Special case: URL type fields with LinkedIn-related context
  if (isLinkedInUrlField || (type === 'url' && (labelText.includes('linkedin') || parentText.includes('linkedin') || 
      labelText.includes('social') || parentText.includes('social') || 
      labelText.includes('profile') || parentText.includes('profile')))) {
    console.log(`🎯 LINKEDIN URL FIELD DETECTED:`, element);
    return true;
  }
  
  // Fallback: If it's a URL field with no name/id and we have a LinkedIn URL to fill,
  // and there are no other obvious URL fields, this might be the LinkedIn field
  if (type === 'url' && !name && !id && !placeholder && !labelText && !parentText) {
    console.log(`🤔 POTENTIAL LINKEDIN FIELD (URL with no attributes):`, element);
    // Check if this is the only URL field or if there are other URL fields that are clearly not LinkedIn
    const allUrlFields = document.querySelectorAll('input[type="url"]');
    if (allUrlFields.length === 1) {
      console.log(`🎯 LINKEDIN FIELD DETECTED (only URL field):`, element);
      return true;
    } else {
      // If there are multiple URL fields, check if others have clear non-LinkedIn purposes
      let hasClearNonLinkedInUrls = false;
      allUrlFields.forEach(urlField => {
        if (urlField !== element) {
          const otherLabel = getLabelText(urlField).toLowerCase();
          const otherParent = getParentText(urlField).toLowerCase();
          if (otherLabel.includes('website') || otherLabel.includes('portfolio') || 
              otherLabel.includes('github') || otherParent.includes('website') || 
              otherParent.includes('portfolio') || otherParent.includes('github')) {
            hasClearNonLinkedInUrls = true;
          }
        }
      });
      
      if (!hasClearNonLinkedInUrls) {
        console.log(`🎯 LINKEDIN FIELD DETECTED (no other clear URL purposes):`, element);
        return true;
      }
    }
  }
  
  return false;
}

// Function to fill any field by looking at all possible attributes
function fillField(value, fieldType) {
  if (!value || value.trim() === '') return false;
  
  console.log(`Looking for ${fieldType} field with value: ${value}`);
  
  // Get ALL form elements
  const allElements = document.querySelectorAll('input, textarea, select');
  console.log(`Found ${allElements.length} form elements`);
  
  let filled = false;
  
  for (const element of allElements) {
    try {
      const name = (element.name || '').toLowerCase();
      const id = (element.id || '').toLowerCase();
      const placeholder = (element.placeholder || '').toLowerCase();
      const type = (element.type || '').toLowerCase();
      
      // Get text from parent elements
      let parentText = '';
      let labelText = '';
      
      if (element.parentElement) {
        parentText = element.parentElement.textContent.toLowerCase();
      }
      
      // Look for associated label
      const label = element.closest('label') || 
                   document.querySelector(`label[for="${element.id}"]`) ||
                   element.parentElement?.querySelector('label');
      if (label) {
        labelText = label.textContent.toLowerCase();
      }
      
      console.log(`Checking element: name="${element.name}", id="${element.id}", type="${element.type}"`);
      console.log(`Parent text: "${parentText.substring(0, 100)}..."`);
      console.log(`Label text: "${labelText}"`);
      
      // Check if this field matches what we're looking for
      let shouldFill = false;
      
      if (fieldType === 'fullName') {
        shouldFill = name.includes('name') || id.includes('name') || placeholder.includes('name') || 
                    labelText.includes('name') || parentText.includes('name') ||
                    placeholder.includes('type here') || placeholder.includes('your name');
      } else if (fieldType === 'firstName') {
        shouldFill = name.includes('first') || id.includes('first') || placeholder.includes('first') || 
                    labelText.includes('first name') || parentText.includes('first name');
      } else if (fieldType === 'lastName') {
        shouldFill = name.includes('last') || id.includes('last') || placeholder.includes('last') || 
                    labelText.includes('last name') || parentText.includes('last name');
      } else if (fieldType === 'email') {
        shouldFill = type === 'email' || name.includes('email') || id.includes('email') || 
                    placeholder.includes('email') || labelText.includes('email');
      } else if (fieldType === 'city') {
        shouldFill = name.includes('city') || id.includes('city') || placeholder.includes('city') || 
                    labelText.includes('city') || parentText.includes('city');
      } else if (fieldType === 'country') {
        // Skip country fields here - handled by special function later
        shouldFill = false;
      } else if (fieldType === 'state') {
        shouldFill = name.includes('state') || id.includes('state') || placeholder.includes('state') || 
                    labelText.includes('state') || parentText.includes('state') || parentText.includes('residence');
      } else if (fieldType === 'salary') {
        shouldFill = name.includes('salary') || name.includes('compensation') || id.includes('salary') || 
                    id.includes('compensation') || placeholder.includes('salary') || placeholder.includes('compensation') ||
                    labelText.includes('salary') || labelText.includes('compensation') || parentText.includes('salary') || parentText.includes('compensation');
      } else if (fieldType === 'startDate') {
        shouldFill = name.includes('start') || name.includes('timeframe') || name.includes('availability') || 
                    id.includes('start') || id.includes('timeframe') || id.includes('availability') ||
                    labelText.includes('start') || labelText.includes('timeframe') || labelText.includes('availability') ||
                    parentText.includes('start') || parentText.includes('timeframe') || parentText.includes('availability');
      } else if (fieldType === 'education') {
        shouldFill = name.includes('education') || name.includes('degree') || id.includes('education') || 
                    id.includes('degree') || labelText.includes('education') || parentText.includes('education');
      } else if (fieldType === 'workAuth') {
        shouldFill = name.includes('authorization') || name.includes('work') || name.includes('legal') || 
                    id.includes('authorization') || id.includes('work') || id.includes('legal') ||
                    labelText.includes('authorization') || labelText.includes('work') || labelText.includes('legal') ||
                    parentText.includes('authorization') || parentText.includes('work') || parentText.includes('legal') ||
                    parentText.includes('visa') || parentText.includes('sponsorship');
      } else if (fieldType === 'location') {
        shouldFill = name.includes('location') || name.includes('live') || name.includes('resident') || 
                    id.includes('location') || id.includes('live') || id.includes('resident') ||
                    labelText.includes('location') || labelText.includes('live') || parentText.includes('location') || parentText.includes('live');
      } else if (fieldType === 'address') {
        shouldFill = name.includes('address') || name.includes('location') || name.includes('where') || 
                    id.includes('address') || id.includes('location') || id.includes('where') ||
                    placeholder.includes('address') || placeholder.includes('location') || placeholder.includes('where') ||
                    labelText.includes('address') || labelText.includes('location') || labelText.includes('where') || 
                    parentText.includes('address') || parentText.includes('location') || parentText.includes('where');
      } else if (fieldType === 'experience') {
        shouldFill = name.includes('experience') || name.includes('years') || id.includes('experience') || 
                    id.includes('years') || labelText.includes('experience') || parentText.includes('experience');
      } else if (fieldType === 'interests') {
        shouldFill = name.includes('interest') || name.includes('expertise') || name.includes('skill') || 
                    name.includes('specialization') || name.includes('passion') || name.includes('focus') ||
                    id.includes('interest') || id.includes('expertise') || id.includes('skill') || 
                    id.includes('specialization') || id.includes('passion') || id.includes('focus') ||
                    placeholder.includes('interest') || placeholder.includes('expertise') || placeholder.includes('skill') ||
                    placeholder.includes('specialization') || placeholder.includes('passion') || placeholder.includes('focus') ||
                    labelText.includes('interest') || labelText.includes('expertise') || labelText.includes('skill') ||
                    labelText.includes('specialization') || labelText.includes('passion') || labelText.includes('focus') ||
                    parentText.includes('interest') || parentText.includes('expertise') || parentText.includes('skill') ||
                    parentText.includes('specialization') || parentText.includes('passion') || parentText.includes('focus');
      } else if (fieldType === 'linkedin') {
        console.log(`🔍 TESTING LINKEDIN DETECTION for element:`, {
          name: element.name,
          id: element.id,
          type: element.type,
          placeholder: element.placeholder,
          className: element.className
        });
        shouldFill = isLinkedInField(element);
        console.log(`🔍 LINKEDIN DETECTION RESULT: ${shouldFill}`);
      } else if (fieldType === 'portfolio') {
        shouldFill = name.includes('portfolio') || name.includes('website') || id.includes('portfolio') || 
                    id.includes('website') || placeholder.includes('portfolio') || placeholder.includes('website') ||
                    labelText.includes('portfolio') || labelText.includes('website') || parentText.includes('portfolio') || parentText.includes('website');
      } else if (fieldType === 'github') {
        shouldFill = name.includes('github') || id.includes('github') || placeholder.includes('github') ||
                    labelText.includes('github') || parentText.includes('github') ||
                    name.includes('gh') || id.includes('gh');
      } else if (fieldType === 'summary') {
        shouldFill = name.includes('summary') || name.includes('cover') || id.includes('summary') || 
                    id.includes('cover') || placeholder.includes('summary') || placeholder.includes('cover') ||
                    labelText.includes('summary') || labelText.includes('cover') || parentText.includes('summary') || parentText.includes('cover');
      } else if (fieldType === 'organizedExample') {
        shouldFill = parentText.includes('organized') || parentText.includes('reliable') || 
                    labelText.includes('organized') || labelText.includes('reliable');
      } else if (fieldType === 'lookerExperience') {
        shouldFill = parentText.includes('looker') || labelText.includes('looker') || 
                    parentText.includes('fluent') || labelText.includes('fluent');
      } else if (fieldType === 'spreadsheetExample') {
        shouldFill = parentText.includes('spreadsheet') || labelText.includes('spreadsheet') || 
                    parentText.includes('excel') || labelText.includes('excel');
      } else if (fieldType === 'visaSponsorship') {
        shouldFill = parentText.includes('visa') || parentText.includes('sponsorship') || 
                    labelText.includes('visa') || labelText.includes('sponsorship');
      } else if (fieldType === 'resume') {
        shouldFill = name.includes('resume') || name.includes('cv') || name.includes('document') ||
                    id.includes('resume') || id.includes('cv') || id.includes('document') ||
                    placeholder.includes('resume') || placeholder.includes('cv') ||
                    labelText.includes('resume') || labelText.includes('cv') ||
                    parentText.includes('resume') || parentText.includes('cv') ||
                    parentText.includes('upload') || parentText.includes('file');
      }
      
      if (shouldFill) {
        console.log(`🎯 FOUND MATCHING FIELD for ${fieldType}:`, element);
        
        // Fill the field
        if (fieldType === 'resume' && element.type === 'file') {
          // Handle file upload for resume
          if (fillResumeField(element, value)) {
            console.log(`✅ FILLED RESUME FIELD: ${value}`);
            filled = true;
          }
        } else if (element.tagName === 'SELECT') {
          // Handle select dropdowns
          const options = element.querySelectorAll('option');
          console.log(`Found ${options.length} options in select`);
          
          for (const option of options) {
            const optionText = option.textContent.toLowerCase().trim();
            const searchValue = value.toLowerCase().trim();
            
            console.log(`Checking option: "${optionText}" against "${searchValue}"`);
            
            // Flexible matching for dropdowns
            if (option.value === value || 
                optionText.includes(searchValue) || 
                searchValue.includes(optionText) ||
                (optionText.includes('yes') && searchValue.includes('yes')) ||
                (optionText.includes('no') && searchValue.includes('no')) ||
                (optionText.includes('authorized') && searchValue.includes('authorized')) ||
                (optionText.includes('legal') && searchValue.includes('legal')) ||
                (optionText.includes('sponsorship') && searchValue.includes('sponsorship')) ||
                (optionText.includes('immediately') && searchValue.includes('immediately')) ||
                (optionText.includes('bachelor') && searchValue.includes('bachelor')) ||
                (optionText.includes('master') && searchValue.includes('master')) ||
                (optionText.includes('phd') && searchValue.includes('phd')) ||
                (optionText.includes('high school') && searchValue.includes('high school')) ||
                // Enhanced country matching - prioritize United States for dropdown selection
                (optionText === 'united states' && (searchValue.includes('united states') || searchValue.includes('usa') || searchValue.includes('us'))) ||
                (optionText === 'united states +1' && (searchValue.includes('united states') || searchValue.includes('usa') || searchValue.includes('us'))) ||
                (optionText === 'usa' && (searchValue.includes('united states') || searchValue.includes('usa') || searchValue.includes('us'))) ||
                (optionText === 'us' && (searchValue.includes('united states') || searchValue.includes('usa') || searchValue.includes('us'))) ||
                (optionText === 'america' && (searchValue.includes('united states') || searchValue.includes('usa') || searchValue.includes('us') || searchValue.includes('america'))) ||
                (optionText === 'united states of america' && (searchValue.includes('united states') || searchValue.includes('usa') || searchValue.includes('us')))) {
              
              element.value = option.value;
              element.dispatchEvent(new Event('change', { bubbles: true }));
              console.log(`✅ FILLED SELECT: ${fieldType} = ${value} (matched: "${optionText}")`);
              filled = true;
              break;
            }
          }
        } else {
          // Handle input fields and textareas with better event handling
          element.focus();
          
          // Clear any existing value first
          element.value = '';
          
          // Set the value
          element.value = value;
          
          // Ensure text is visible
          ensureTextVisibility(element);
          
          // Trigger events in the correct order
          element.dispatchEvent(new Event('focus', { bubbles: true }));
          element.dispatchEvent(new Event('input', { bubbles: true }));
          element.dispatchEvent(new Event('change', { bubbles: true }));
          element.dispatchEvent(new Event('blur', { bubbles: true }));
          
          // Force the value to stick and maintain visibility
          setTimeout(() => {
            if (element.value !== value) {
              element.value = value;
              ensureTextVisibility(element);
              element.dispatchEvent(new Event('input', { bubbles: true }));
            }
          }, 100);
          
          console.log(`✅ FILLED INPUT: ${fieldType} = ${value}`);
          filled = true;
        }
        
        // Highlight the filled field
        element.style.outline = '3px solid #4CAF50';
        element.style.backgroundColor = '#e8f5e8';
        
        break; // Stop after finding and filling the first match
      }
    } catch (error) {
      console.log(`Error processing field:`, error);
    }
  }
  
  return filled;
}

// Smart field detection and filling
function fillCheckboxRadio(type, value) {
  console.log(`🎯 Looking for ${type} authorization fields...`);
  
  let filled = false;
  
  // Step 1: First, analyze and count all form elements
  const analysis = analyzeFormElements();
  console.log(`📊 Form Analysis Results:`);
  console.log(`   - Text inputs: ${analysis.textInputs}`);
  console.log(`   - Dropdowns: ${analysis.dropdowns}`);
  console.log(`   - Checkboxes: ${analysis.checkboxes}`);
  console.log(`   - Radio buttons: ${analysis.radioButtons}`);
  console.log(`   - Select elements: ${analysis.selectElements}`);
  console.log(`   - Clickable elements: ${analysis.clickableElements}`);
  console.log(`   - ARROWS FOUND: ${analysis.arrows}`);
  
  // Confirm methods available for each element type
  console.log(`🔧 Available Methods:`);
  console.log(`   - Text inputs → fillTextInput()`);
  console.log(`   - Dropdowns → fillDropdown()`);
  console.log(`   - Checkboxes → fillCheckbox()`);
  console.log(`   - Radio buttons → fillRadio()`);
  console.log(`   - Select elements → fillSelect()`);
  console.log(`   - Clickable elements → fillClickableElement()`);
  console.log(`   - Arrows → fillDropdown() (clicks arrow first)`);
  
  // Step 2: Find all possible elements (inputs, selects, divs, etc.)
  const allElements = document.querySelectorAll('input, select, div, span, label, [role="radio"]');
  console.log(`Found ${allElements.length} total elements to check`);
  
  for (const element of allElements) {
    // Step 3: Detect element type
    const elementType = detectElementType(element);
    console.log(`Element type: ${elementType}`);
    
    // Step 4: Get context text
    const container = element.closest('div, fieldset, form, section') || element.parentElement;
    const containerText = container ? container.textContent.toLowerCase() : '';
    const elementText = element.textContent ? element.textContent.toLowerCase() : '';
    
    // Step 5: Check if this matches our target field
    if (isTargetField(element, containerText, elementText, type)) {
      console.log(`Found ${type} field with element type: ${elementType}`);
      
      // Step 6: Use appropriate method based on element type
      if (elementType === 'checkbox') {
        filled = fillCheckbox(element, value, type);
      } else if (elementType === 'radio') {
        filled = fillRadio(element, value, type);
      } else if (elementType === 'select') {
        filled = fillSelect(element, value, type);
      } else if (elementType === 'dropdown') {
        filled = fillDropdown(element, value, type);
  } else if (elementType === 'text') {
        filled = fillTextInput(element, value, type);
  } else if (elementType === 'div' || elementType === 'span' || elementType === 'radio-role') {
        filled = fillClickableElement(element, value, type);
      }
      
      if (filled) {
        console.log(`✅ Successfully filled ${type} field using ${elementType} method`);
        break;
      }
    }
  }
  
  return filled;
}

// Analyze form elements to understand the page structure
function analyzeFormElements() {
  const analysis = {
    textInputs: 0,
    dropdowns: 0,
    checkboxes: 0,
    radioButtons: 0,
    selectElements: 0,
    clickableElements: 0,
    arrows: 0
  };
  
  // Count text inputs
  analysis.textInputs = document.querySelectorAll('input[type="text"], input[type="email"], input[type="tel"], input[type="url"], input:not([type])').length;
  
  // Count checkboxes
  analysis.checkboxes = document.querySelectorAll('input[type="checkbox"]').length;
  
  // Count radio buttons
  analysis.radioButtons = document.querySelectorAll('input[type="radio"]').length;
  
  // Count select elements
  analysis.selectElements = document.querySelectorAll('select').length;
  
  // Count ALL arrows on the page
  const arrowSelectors = [
    '▼', '▼', '▼', '▼', '▼', '▼', '▼', '▼', '▼', '▼',
    '.arrow', '[class*="arrow"]', '[class*="dropdown"]', '[class*="select"]',
    'svg', 'i[class*="arrow"]', 'i[class*="down"]', 'i[class*="chevron"]',
    'button[class*="arrow"]', 'button[class*="dropdown"]', 'button[class*="select"]'
  ];
  
  for (const selector of arrowSelectors) {
    if (selector.length === 1) {
      // Unicode arrow character
      const elements = document.querySelectorAll('*');
      for (const element of elements) {
        if (element.innerHTML.includes(selector) || element.textContent.includes(selector)) {
          analysis.arrows++;
        }
      }
    } else {
      // CSS selector
      analysis.arrows += document.querySelectorAll(selector).length;
    }
  }
  
  // Count dropdowns (divs/spans with arrows)
  const allDivs = document.querySelectorAll('div, span');
  for (const div of allDivs) {
    const hasArrow = div.innerHTML.includes('▼') || 
                    div.innerHTML.includes('▼') || 
                    div.innerHTML.includes('▼') ||
                    div.innerHTML.includes('▼') ||
                    div.querySelector('.arrow') ||
                    div.querySelector('[class*="arrow"]') ||
                    div.querySelector('[class*="dropdown"]') ||
                    div.querySelector('[class*="select"]');
    
    if (hasArrow) {
      analysis.dropdowns++;
    }
  }
  
  // Count clickable elements
  analysis.clickableElements = document.querySelectorAll('div[onclick], span[onclick], div[role="button"], span[role="button"]').length;
  
  return analysis;
}

// Detect what type of element this is
function detectElementType(element) {
  if (element.tagName === 'INPUT') {
    return element.type || 'text';
  } else if (element.tagName === 'SELECT') {
    return 'select';
  } else if (element.tagName === 'DIV' || element.tagName === 'SPAN') {
    // Check if it has dropdown arrow (▼, ▼, ▼, etc.)
    const hasArrow = element.innerHTML.includes('▼') || 
                    element.innerHTML.includes('▼') || 
                    element.innerHTML.includes('▼') ||
                    element.innerHTML.includes('▼') ||
                    element.querySelector('.arrow') ||
                    element.querySelector('[class*="arrow"]') ||
                    element.querySelector('[class*="dropdown"]') ||
                    element.querySelector('[class*="select"]');
    
    if (hasArrow) {
      return 'dropdown';
    }
    return element.tagName.toLowerCase();
  } else if (element.tagName === 'LABEL') {
    return 'label';
  } else if (element.getAttribute && element.getAttribute('role') === 'radio') {
    return 'radio-role';
  }
  return 'unknown';
}

// Check if this element matches our target field
function isTargetField(element, containerText, elementText, type) {
  const nycRx = /(nyc|new\s+york|hybrid)/i;
  if (type === 'work') {
    return containerText.includes('authorized') || 
           containerText.includes('legal') || 
           containerText.includes('work') ||
           elementText.includes('authorized') ||
           elementText.includes('legal');
  } else if (type === 'visa') {
    return containerText.includes('sponsor') || 
           containerText.includes('visa') || 
           containerText.includes('immigration') ||
           elementText.includes('sponsor') ||
           elementText.includes('visa');
  } else if (type === 'nycOffice') {
    return nycRx.test(containerText) || nycRx.test(elementText);
  }
  return false;
}

// Fill checkbox element
function fillCheckbox(element, value, type) {
  console.log(`Filling checkbox for ${type}`);
  const label = element.closest('label') || element.nextElementSibling;
  const labelText = label ? label.textContent.toLowerCase() : '';
  
  if (value.toLowerCase().includes('yes') || value.toLowerCase().includes('true')) {
    if (labelText.includes('yes')) {
      element.click();
      element.style.outline = '3px solid #4CAF50';
      return true;
    }
  } else {
    if (labelText.includes('no')) {
      element.click();
      element.style.outline = '3px solid #4CAF50';
      return true;
    }
  }
  return false;
}

// Fill radio button element
function fillRadio(element, value, type) {
  console.log(`Filling radio for ${type}`);
  const label = element.closest('label') || element.nextElementSibling;
  const labelText = label ? label.textContent.toLowerCase() : '';
  
  if (value.toLowerCase().includes('yes') || value.toLowerCase().includes('true')) {
    if (labelText.includes('yes')) {
      element.click();
      element.style.outline = '3px solid #4CAF50';
      return true;
    }
  } else {
    if (labelText.includes('no')) {
      element.click();
      element.style.outline = '3px solid #4CAF50';
      return true;
    }
  }
  return false;
}

// Fill select dropdown element
function fillSelect(element, value, type) {
  console.log(`Filling select for ${type}`);
  const options = element.querySelectorAll('option');
  
  for (const option of options) {
    const optionText = option.textContent.toLowerCase();
    if (value.toLowerCase().includes('yes') || value.toLowerCase().includes('true')) {
      if (optionText.includes('yes')) {
        element.value = option.value;
        element.dispatchEvent(new Event('change', { bubbles: true }));
        element.style.outline = '3px solid #4CAF50';
        return true;
      }
    } else {
      if (optionText.includes('no')) {
        element.value = option.value;
        element.dispatchEvent(new Event('change', { bubbles: true }));
        element.style.outline = '3px solid #4CAF50';
        return true;
      }
    }
  }
  return false;
}

// Fill text input element
function fillTextInput(element, value, type) {
  console.log(`Filling text input for ${type}`);
  element.focus();
  element.value = value;
  
  // Ensure text is visible
  ensureTextVisibility(element);
  
  element.dispatchEvent(new Event('input', { bubbles: true }));
  element.dispatchEvent(new Event('change', { bubbles: true }));
  element.style.outline = '3px solid #4CAF50';
  return true;
}

// Fill dropdown element (div/span with arrow)
function fillDropdown(element, value, type) {
  console.log(`🎯 Smart dropdown filling for ${type}`);
  
  // Step 1: Look for arrow button first - be smart about it
  const arrowSelectors = [
    'button[class*="arrow"]',
    'button[class*="dropdown"]', 
    'button[class*="select"]',
    '.arrow',
    '.dropdown-arrow',
    '.select-arrow',
    '[class*="arrow"]',
    'svg',
    'i[class*="arrow"]',
    'i[class*="down"]',
    'i[class*="chevron"]'
  ];
  
  let arrow = null;
  for (const selector of arrowSelectors) {
    arrow = element.querySelector(selector);
    if (arrow) {
      console.log(`✅ Found arrow button with selector: ${selector}`);
      break;
    }
  }
  
  if (arrow) {
    console.log(`🖱️ Clicking arrow button to open dropdown`);
    arrow.click();
    element.style.outline = '3px solid #4CAF50';
  } else {
    console.log(`⚠️ No arrow found, clicking the whole element`);
    element.click();
    element.style.outline = '3px solid #4CAF50';
  }
  
  // Step 2: Wait for dropdown to open - be patient
  setTimeout(() => {
    console.log(`⏳ Looking for dropdown options...`);
    
    // Step 3: Look for dropdown options with multiple strategies
    const optionSelectors = [
      'div[role="option"]',
      'li[role="option"]', 
      '.dropdown-option',
      '.select-option',
      '[class*="option"]',
      '.menu-item',
      '.dropdown-item',
      'li',
      'div[class*="item"]'
    ];
    
    let allOptions = [];
    for (const selector of optionSelectors) {
      const options = document.querySelectorAll(selector);
      if (options.length > 0) {
        console.log(`Found ${options.length} options with selector: ${selector}`);
        allOptions = Array.from(options);
        break;
      }
    }
    
    console.log(`📋 Total options found: ${allOptions.length}`);
    
    // Step 4: Find and click the right option - be smart about matching
    for (const option of allOptions) {
      const optionText = option.textContent.toLowerCase().trim();
      console.log(`🔍 Checking option: "${optionText}"`);
      
      if (value.toLowerCase().includes('yes') || value.toLowerCase().includes('true')) {
        if (optionText.includes('yes') || 
            optionText.includes('authorized') || 
            optionText.includes('legal') ||
            optionText.includes('have legal right')) {
          console.log(`✅ Clicking YES option: ${option.textContent}`);
          option.click();
          option.style.outline = '3px solid #4CAF50';
          return true;
        }
      } else {
        if (optionText.includes('no') || 
            optionText.includes('not require') || 
            optionText.includes('do not') ||
            optionText.includes('do not require')) {
          console.log(`✅ Clicking NO option: ${option.textContent}`);
          option.click();
          option.style.outline = '3px solid #4CAF50';
          return true;
        }
      }
    }
    
    console.log(`❌ No matching option found`);
  }, 800); // Wait longer for dropdown to fully open
  
  return true; // Return true because we clicked the dropdown
}

// Fill clickable div/span element
function fillClickableElement(element, value, type) {
  console.log(`Filling clickable element for ${type}`);
  const elementText = element.textContent.toLowerCase();
  const aria = ((element.getAttribute && (element.getAttribute('aria-label') || '')) || '').toLowerCase();
  const combined = `${elementText} ${aria}`;
  
  if (value.toLowerCase().includes('yes') || value.toLowerCase().includes('true')) {
    if (combined.includes('yes') || combined.includes('i can') || combined.includes('able')) {
      element.click();
      element.style.outline = '3px solid #4CAF50';
      return true;
    }
  } else {
    if (combined.includes('no') || combined.includes("can't") || combined.includes('cannot')) {
      element.click();
      element.style.outline = '3px solid #4CAF50';
      return true;
    }
  }
  return false;
}

// Robust radio selection by question text (works with label, role=radio, or input+label)
function clickRadioByQuestion(questionRx, wantYes) {
  try {
    const groups = [...document.querySelectorAll('fieldset,section,form,div')]
      .filter(n => questionRx.test(n.textContent || '') && n.querySelector('input[type="radio"],[role="radio"]'));
    if (!groups.length) return false;
    const group = groups.sort((a,b)=>((a.textContent||'').length - (b.textContent||'').length))[0];

    const token = wantYes ? 'yes' : 'no';
    const yesNoValRx = wantYes ? /(yes|true)/i : /(no|false)/i;
    const tokenRx = new RegExp(`(^|\\s)${token}(\\s|$)`, 'i');

    // 1) label text
    let target = [...group.querySelectorAll('label')].find(l => tokenRx.test(l.textContent || ''));
    // 2) role=radio aria/text
    if (!target) {
      target = [...group.querySelectorAll('[role="radio"]')]
        .find(e => tokenRx.test((e.getAttribute('aria-label') || e.textContent || '')));
    }
    // 3) inputs by value or label[for]
    if (!target) {
      const input = [...group.querySelectorAll('input[type="radio"]')].find(r =>
        yesNoValRx.test(r.value || '') || tokenRx.test((document.querySelector(`label[for="${r.id}"]`)?.textContent || ''))
      );
      if (input) target = input;
    }
    if (!target) return false;

    target.scrollIntoView({ block: 'center' });
    ['pointerdown','mousedown','mouseup','click'].forEach(t =>
      target.dispatchEvent(new MouseEvent(t, { bubbles: true }))
    );
    console.log('NYC> clicked', token, target);
    return true;
  } catch (e) { console.log('NYC> radio click error', e); }
  return false;
}

// Targeted handler for: "Are you able to work from our NYC office Tuesday - Thursday?"
function clickNycTueThuRadio(wantYes) {
  try {
    const questionRx = /(work\s*from\s*our\s*nyc\s*office)[\s\S]*tuesday[\s\S]*thursday/i;
    const isYes = t => (/\byes\b/i).test(t);
    const isNo  = t => (/\bno\b/i).test(t);
    const txt = el => ((el && (el.getAttribute && el.getAttribute('aria-label'))) || el?.textContent || '').trim();
    const click = el => el && ['pointerdown','mousedown','mouseup','click'].forEach(type => el.dispatchEvent(new MouseEvent(type, { bubbles: true })));

    const anchors = Array.from(document.querySelectorAll('label,legend,h1,h2,h3,h4,h5,h6,div,section,p'));
    const anchor = anchors.find(n => questionRx.test(n.textContent || ''));
    if (!anchor) return false;

    const roots = [anchor, anchor.parentElement, anchor.nextElementSibling, anchor.closest('div,section,fieldset,form')].filter(Boolean);

    // 1) Native inputs
    for (const root of roots) {
      const radios = Array.from(root.querySelectorAll('input[type="radio"]'));
      if (radios.length) {
        for (const r of radios) {
          const lbl = r.id ? (document.querySelector(`label[for="${r.id}"]`)?.textContent || '') : (r.closest('label')?.textContent || '');
          const val = `${r.value || ''} ${lbl}`;
          if ((wantYes && isYes(val)) || (!wantYes && isNo(val))) { click(r); return true; }
        }
      }
    }

    // 2) Segmented or role-based radios (Ashby-style)
    const roleSel = '[role="radio"],[role="option"],button,div,span,label';
    for (const root of roots) {
      const opts = Array.from(root.querySelectorAll(roleSel)).filter(e => isYes(txt(e)) || isNo(txt(e)));
      const pick = wantYes ? opts.find(e => isYes(txt(e))) : opts.find(e => isNo(txt(e)));
      if (pick) { click(pick.closest('[role="radio"],[role="option"],button,label,div,span') || pick); return true; }
    }

    return false;
  } catch (_) { return false; }
}

// Robust target: "Are you legally authorized to work in the United States?"
function clickLegalAuthorizedRadio(wantYes) {
  try {
    const questionRx = /are\s+you\s+legally\s+authorized[\s\S]*united\s*states/i;
    const yesRx = /^\s*yes\s*$/i; const noRx = /^\s*no\s*$/i;
    const getTxt = el => ((el?.getAttribute && el.getAttribute('aria-label')) || el?.textContent || '').trim();
    const click = el => { if (!el) return false; el.scrollIntoView?.({ block: 'center' }); ['pointerdown','mousedown','mouseup','click'].forEach(t=>el.dispatchEvent(new MouseEvent(t,{bubbles:true}))); if (el.type==='radio'){ el.checked=true; el.dispatchEvent(new Event('input',{bubbles:true})); el.dispatchEvent(new Event('change',{bubbles:true})); } return true; };
    const containers = Array.from(document.querySelectorAll('fieldset,section,div,form'))
      .filter(n => questionRx.test(n.textContent || ''))
      .sort((a,b)=> (a.textContent||'').length - (b.textContent||'').length);
    const root = containers[0];
    if (!root) return false;
    for (const r of Array.from(root.querySelectorAll('input[type="radio"]'))) {
      const lbl = (r.id && (document.querySelector(`label[for="${r.id}"]`)?.textContent||'')) || (r.closest('label')?.textContent||'');
      const combined = `${r.value||''} ${lbl}`;
      if ((wantYes && yesRx.test(combined)) || (!wantYes && noRx.test(combined))) return click(r);
    }
    const roleSel='[role="radio"],[role="option"],button,label,div,span';
    const opts = Array.from(root.querySelectorAll(roleSel)).filter(e => yesRx.test(getTxt(e)) || noRx.test(getTxt(e)));
    const pick = wantYes ? opts.find(e => yesRx.test(getTxt(e))) : opts.find(e => noRx.test(getTxt(e)));
    return pick ? click(pick.closest('[role="radio"],[role="option"],button,label,div,span')||pick) : false;
  } catch(_) { return false; }
}

// Robust target: sponsorship question (H1B/visa status)
function clickSponsorshipRadio(wantYes) {
  try {
    const questionRx = /(require\s+sponsorship|visa\s*status|h-?1b)/i;
    const yesRx = /^\s*yes\s*$/i; const noRx = /^\s*no\s*$/i;
    const getTxt = el => ((el?.getAttribute && el.getAttribute('aria-label')) || el?.textContent || '').trim();
    const click = el => { if (!el) return false; el.scrollIntoView?.({ block: 'center' }); ['pointerdown','mousedown','mouseup','click'].forEach(t=>el.dispatchEvent(new MouseEvent(t,{bubbles:true}))); if (el.type==='radio'){ el.checked=true; el.dispatchEvent(new Event('input',{bubbles:true})); el.dispatchEvent(new Event('change',{bubbles:true})); } return true; };
    const containers = Array.from(document.querySelectorAll('fieldset,section,div,form'))
      .filter(n => questionRx.test(n.textContent || ''))
      .sort((a,b)=> (a.textContent||'').length - (b.textContent||'').length);
    const root = containers[0];
    if (!root) return false;
    for (const r of Array.from(root.querySelectorAll('input[type="radio"]'))) {
      const lbl = (r.id && (document.querySelector(`label[for="${r.id}"]`)?.textContent||'')) || (r.closest('label')?.textContent||'');
      const combined = `${r.value||''} ${lbl}`;
      if ((wantYes && yesRx.test(combined)) || (!wantYes && noRx.test(combined))) return click(r);
    }
    const roleSel='[role="radio"],[role="option"],button,label,div,span';
    const opts = Array.from(root.querySelectorAll(roleSel)).filter(e => yesRx.test(getTxt(e)) || noRx.test(getTxt(e)));
    const pick = wantYes ? opts.find(e => yesRx.test(getTxt(e))) : opts.find(e => noRx.test(getTxt(e)));
    return pick ? click(pick.closest('[role="radio"],[role="option"],button,label,div,span')||pick) : false;
  } catch(_) { return false; }
}

// Robust target: located in or willing to relocate to NYC
function clickRelocateNYCRadio(wantYes) {
  try {
    const questionRx = /(located[\s\S]*relocate[\s\S]*nyc|willing[\s\S]*relocate[\s\S]*nyc|relocate\s*to\s*nyc)/i;
    const yesRx = /^\s*yes\s*$/i; const noRx = /^\s*no\s*$/i;
    const getTxt = el => ((el?.getAttribute && el.getAttribute('aria-label')) || el?.textContent || '').trim();
    const click = el => { if (!el) return false; el.scrollIntoView?.({ block: 'center' }); ['pointerdown','mousedown','mouseup','click'].forEach(t=>el.dispatchEvent(new MouseEvent(t,{bubbles:true}))); if (el.type==='radio'){ el.checked=true; el.dispatchEvent(new Event('input',{bubbles:true})); el.dispatchEvent(new Event('change',{bubbles:true})); } return true; };
    const containers = Array.from(document.querySelectorAll('fieldset,section,div,form'))
      .filter(n => questionRx.test(n.textContent || ''))
      .sort((a,b)=> (a.textContent||'').length - (b.textContent||'').length);
    const root = containers[0];
    if (!root) return false;
    for (const r of Array.from(root.querySelectorAll('input[type="radio"]'))) {
      const lbl = (r.id && (document.querySelector(`label[for="${r.id}"]`)?.textContent||'')) || (r.closest('label')?.textContent||'');
      const combined = `${r.value||''} ${lbl}`;
      if ((wantYes && yesRx.test(combined)) || (!wantYes && noRx.test(combined))) return click(r);
    }
    const roleSel='[role="radio"],[role="option"],button,label,div,span';
    const opts = Array.from(root.querySelectorAll(roleSel)).filter(e => yesRx.test(getTxt(e)) || noRx.test(getTxt(e)));
    const pick = wantYes ? opts.find(e => yesRx.test(getTxt(e))) : opts.find(e => noRx.test(getTxt(e)));
    return pick ? click(pick.closest('[role="radio"],[role="option"],button,label,div,span')||pick) : false;
  } catch(_) { return false; }
}

// Helper: try to set a native <select> inside root to Yes/No
function setYesNoSelectInRoot(root, wantYes) {
  try {
    const selects = Array.from(root.querySelectorAll('select'));
    const yesRx = /\byes\b/i; const noRx = /\bno\b/i;
    for (const sel of selects) {
      const opts = Array.from(sel.querySelectorAll('option'));
      const match = wantYes ? opts.find(o => yesRx.test(o.textContent||'')) : opts.find(o => noRx.test(o.textContent||''));
      if (match) {
        sel.value = match.value;
        sel.dispatchEvent(new Event('change', { bubbles: true }));
        return true;
      }
    }
  } catch(_) {}
  return false;
}

// Stabilizer: re-assert Yes/No selection a few times to survive re-renders
function assertYesNoByQuestion(questionRx, wantYes) {
  const attempts = [0, 150, 600];
  attempts.forEach(delay => {
    setTimeout(() => {
      try {
        // Find the smallest container with the question
        const containers = Array.from(document.querySelectorAll('fieldset,section,div,form'))
          .filter(n => questionRx.test(n.textContent || ''))
          .sort((a,b)=> (a.textContent||'').length - (b.textContent||'').length);
        const root = containers[0];
        if (!root) return;
        // Try radios/role
        const clicked = clickRadioByQuestion(questionRx, wantYes);
        // Try native select within the same root
        if (!clicked) setYesNoSelectInRoot(root, wantYes);
      } catch(_) {}
    }, delay);
  });
}

// Human-like authentication check
async function waitForAuthentication() {
  console.log('⏳ Waiting for user authentication...');
  
  // Wait for page to be fully loaded
  await new Promise(resolve => {
    if (document.readyState === 'complete') {
      resolve();
    } else {
      window.addEventListener('load', resolve);
    }
  });
  
  // Wait a bit more for any authentication to complete
  await new Promise(resolve => setTimeout(resolve, 2000));
  
  console.log('✅ Page loaded and ready for human-like interaction');
}

// Main fill function - uses ONLY local storage
async function fillNow() {
  console.log('=== STARTING ULTRA AGGRESSIVE FILL ===');
  
  try {
    // Enhanced Chrome API availability check
    console.log('Checking Chrome API availability...');
    console.log('typeof chrome:', typeof chrome);
    console.log('chrome object:', chrome);
    
    if (typeof chrome === 'undefined') {
      console.error('❌ Chrome APIs not available - this script must run in a Chrome extension context');
      showNotification('❌ Chrome APIs not available. Please ensure this is running as a Chrome extension.', 'error');
      return;
    }
    
    console.log('Chrome object available, checking storage...');
    console.log('chrome.storage:', chrome.storage);
    console.log('chrome.storage.local:', chrome.storage?.local);
    
    if (!chrome.storage) {
      console.error('❌ Chrome storage API not available');
      showNotification('❌ Storage API not available. Please check extension permissions.', 'error');
      return;
    }
    
    if (!chrome.storage.local) {
      console.error('❌ Chrome storage.local not available');
      showNotification('❌ Storage local API not available. Please check extension permissions.', 'error');
      return;
    }
    
    console.log('✅ All Chrome APIs available, proceeding with fill...');
    
    // Debug: Analyze page structure first
    console.log('🔍 ANALYZING PAGE STRUCTURE...');
    const allElements = document.querySelectorAll('input, textarea, select');
    console.log(`Found ${allElements.length} form elements on this page`);
    
    if (allElements.length === 0) {
      console.log('⚠️ No form elements found - this might not be a form page');
      showNotification('⚠️ No form fields detected on this page', 'warning');
      return;
    }
    
    // Human-like behavior: Wait for authentication
    await waitForAuthentication();
    
    // Get saved profile from local storage ONLY
    console.log('Attempting to load profile from Chrome storage...');
    const result = await chrome.storage.local.get(['jobProfile']);
    const profile = result.jobProfile || {};
    console.log('Profile loaded successfully from Chrome storage');
    
    console.log('Profile loaded for filling:', {
      hasProfile: !!profile,
      hasResumeFileName: !!profile.resumeFileName,
      hasResumeFileData: !!profile.resumeFileData,
      resumeFileName: profile.resumeFileName || 'none',
      allProfileKeys: Object.keys(profile)
    });
    
    console.log('Profile loaded from local storage:', profile);
    
    if (Object.keys(profile).length === 0) {
      console.log('No profile found in local storage');
      showNotification('❌ No profile found. Please save your profile first.', 'error');
      return;
    }
    
    let filledCount = 0;
    
    // Fill name field first (combine first + last name)
    if (profile.firstName || profile.lastName) {
      const fullName = `${profile.firstName || ''} ${profile.lastName || ''}`.trim();
      if (fullName && fillField(fullName, 'fullName')) {
        filledCount++;
        console.log(`✅ Filled full name: ${fullName}`);
      }
    }
    
    // Fill each field type
    const fieldMappings = [
      { key: 'firstName', type: 'firstName' },
      { key: 'lastName', type: 'lastName' },
      { key: 'email', type: 'email' },
      { key: 'city', type: 'city' },
      { key: 'country', type: 'country' },
      { key: 'state', type: 'state' },
      { key: 'salary', type: 'salary' },
      { key: 'startDate', type: 'startDate' },
      { key: 'education', type: 'education' },
      { key: 'workAuth', type: 'workAuth' },
      { key: 'location', type: 'location' },
      { key: 'address', type: 'address' },
      { key: 'experience', type: 'experience' },
      { key: 'linkedin', type: 'linkedin' },
      { key: 'github', type: 'github' },
      { key: 'portfolio', type: 'portfolio' },
      { key: 'summary', type: 'summary' },
      { key: 'interests', type: 'interests' },
      { key: 'organizedExample', type: 'organizedExample' },
      { key: 'lookerExperience', type: 'lookerExperience' },
      { key: 'spreadsheetExample', type: 'spreadsheetExample' },
      { key: 'visaSponsorship', type: 'visaSponsorship' },
      { key: 'resume', type: 'resume' }
    ];
    
    for (const mapping of fieldMappings) {
      if (profile[mapping.key]) {
        console.log(`Attempting to fill ${mapping.type} with: ${profile[mapping.key]}`);
        if (mapping.type === 'linkedin') {
          console.log(`🔍 LINKEDIN FILL ATTEMPT - Value: ${profile[mapping.key]}`);
          // Additional debugging for LinkedIn fields
          const allElements = document.querySelectorAll('input, textarea, select');
          console.log(`🔍 Found ${allElements.length} total form elements`);
          
          // Check for URL type elements specifically
          const urlElements = document.querySelectorAll('input[type="url"]');
          console.log(`🔍 Found ${urlElements.length} URL type elements:`, urlElements);
          
          urlElements.forEach((el, index) => {
            console.log(`🔍 URL Element ${index}:`, {
              name: el.name,
              id: el.id,
              placeholder: el.placeholder,
              className: el.className,
              labelText: getLabelText(el),
              parentText: getParentText(el)
            });
          });
        } else if (mapping.type === 'interests') {
          console.log(`🔍 INTERESTS FILL ATTEMPT - Value: ${profile[mapping.key]}`);
          // Additional debugging for interests fields
          const allElements = document.querySelectorAll('input, textarea, select');
          console.log(`🔍 Found ${allElements.length} total form elements`);
          
          // Check for textarea elements specifically (interests are often textareas)
          const textareaElements = document.querySelectorAll('textarea');
          console.log(`🔍 Found ${textareaElements.length} textarea elements:`, textareaElements);
          
          textareaElements.forEach((el, index) => {
            console.log(`🔍 Textarea Element ${index}:`, {
              name: el.name,
              id: el.id,
              placeholder: el.placeholder,
              className: el.className,
              labelText: getLabelText(el),
              parentText: getParentText(el)
            });
          });
        }
        
        if (fillField(profile[mapping.key], mapping.type)) {
          filledCount++;
          console.log(`✅ Successfully filled ${mapping.type}`);
        } else {
          console.log(`❌ Failed to fill ${mapping.type}`);
        }
      }
    }

    // Explicit GitHub fallback: target common Ashby-style fields by exact id/name and case-insensitive contains
    if (profile.github) {
      try {
        const ghSelectors = [
          'input#GitHub',
          'input[name="GitHub"]',
          'input[id*="github" i]',
          'input[name*="github" i]',
          'input[placeholder*="github" i]'
        ];
        let gh = null;
        for (const sel of ghSelectors) {
          gh = document.querySelector(sel);
          if (gh) break;
        }
        // Label-based fallback (Ashby): find label text 'GitHub' and associated input
        if (!gh) {
          const labels = Array.from(document.querySelectorAll('label'));
          const ghLabel = labels.find(l => (l.getAttribute('for') === 'GitHub') || /\bgithub\b/i.test(l.textContent || ''));
          if (ghLabel) {
            if (ghLabel.getAttribute('for')) {
              gh = document.getElementById(ghLabel.getAttribute('for'));
            }
            if (!gh) {
              const containers = [ghLabel, ghLabel.parentElement, ghLabel.nextElementSibling, ghLabel.closest('div')].filter(Boolean);
              for (const c of containers) {
                const candidate = c && c.querySelector && c.querySelector('input');
                if (candidate) { gh = candidate; break; }
              }
            }
          }
        }
        if (gh) {
          // Use shadowRoot-aware input if present (some Ashby inputs render inside shadow DOM)
          const shadowInput = gh.shadowRoot ? gh.shadowRoot.querySelector('input') : null;
          const target = shadowInput || gh;
          const ghValue = normalizeGithubUrl(profile.github);
          typeLikeHuman(target, ghValue);
          ensureTextVisibility(gh);
          target.style.outline = '3px solid #4CAF50';
          target.style.backgroundColor = '#e8f5e8';
          filledCount++;
          console.log('✅ GitHub filled via explicit fallback');
        } else {
          console.log('ℹ️ GitHub explicit fallback: no matching element found');
        }
      } catch (e) {
        console.log('GitHub explicit fallback error:', e);
      }
    }
    
    // SPECIAL ULTRA AGGRESSIVE PHONE FILL
    if (profile.phone) {
      console.log('🚀 ULTRA AGGRESSIVE PHONE FILL STARTING...');
      const formattedPhone = formatPhoneNumber(profile.phone);
      console.log(`Original phone: ${profile.phone}, Formatted: ${formattedPhone}`);
      
      // Try both original and formatted versions
      if (fillPhoneField(profile.phone) || fillPhoneField(formattedPhone)) {
        filledCount++;
        console.log(`✅ ULTRA AGGRESSIVE filled phone field`);
      } else {
        console.log(`❌ ULTRA AGGRESSIVE failed to fill phone field`);
      }
    }
    
    // SPECIAL ULTRA AGGRESSIVE COUNTRY FILL - HUMAN-LIKE INTERACTION
    if (profile.country) {
      console.log('🚀 ULTRA AGGRESSIVE COUNTRY FILL STARTING...');
      console.log(`Country value: ${profile.country}`);
      
      // Find country dropdown
      const countrySelectors = [
        'select[name*="country"]', 'select[id*="country"]',
        'input[name*="country"]', 'input[id*="country"]',
        '[data-testid*="country"]', '[data-qa*="country"]'
      ];
      
      let countryElement = null;
      for (const selector of countrySelectors) {
        countryElement = document.querySelector(selector);
        if (countryElement) {
          console.log(`Found country element with selector: ${selector}`);
          break;
        }
      }
      
      if (countryElement) {
        console.log('🎯 Found country element, simulating human behavior...');
        
        // Human-like behavior: click to open dropdown
        countryElement.focus();
        countryElement.click();
        
        // Wait for dropdown to open
        setTimeout(() => {
          // Find United States option
          const options = countryElement.querySelectorAll('option');
          let usOption = null;
          
          console.log(`Found ${options.length} options in dropdown:`);
          for (const option of options) {
            console.log(`- "${option.textContent}"`);
          }
          
          for (const option of options) {
            const text = option.textContent.toLowerCase();
            console.log(`Checking option: "${option.textContent}"`);
            
            // Look for United States with multiple patterns
            if (text.includes('united states') || 
                text.includes('usa') || 
                text.includes('us +1') ||
                text.includes('america') ||
                (text.includes('+1') && !text.includes('canada'))) {
              usOption = option;
              console.log(`✅ Found United States: ${option.textContent}`);
              break;
            }
          }
          
          if (usOption) {
            // Human-like: click the option
            usOption.click();
            countryElement.style.outline = '3px solid #4CAF50';
            countryElement.style.backgroundColor = '#e8f5e8';
            filledCount++;
            console.log(`✅ CONFIRMED: Selected ${usOption.textContent}`);
          } else {
            console.log(`❌ United States not found in dropdown`);
            console.log(`Available options: ${Array.from(options).map(o => o.textContent).join(', ')}`);
          }
        }, 300);
      } else {
        console.log(`❌ Could not find country dropdown element`);
      }
    }
    
    // SPECIAL ULTRA AGGRESSIVE CHECKBOX/RADIO FILL
    if (profile.workAuth) {
      console.log('🚀 ULTRA AGGRESSIVE WORK AUTH FILL STARTING...');
      if (fillCheckboxRadio('work', profile.workAuth)) {
        filledCount++;
        console.log(`✅ ULTRA AGGRESSIVE filled work authorization`);
      }
    }

    // NYC/hybrid radio via widget (nycOffice) - isolate side effects
    const snapshot = snapshotTextInputs();
    const nycWantYes = (profile.nycOffice || '').toLowerCase().startsWith('y') || !profile.nycOffice;
    // Legal authorization (only if provided in widget)
    const workPref = (profile.workAuth || '').toLowerCase();
    if (workPref) {
      const legalYes = workPref.includes('legal') || workPref.startsWith('yes');
      clickLegalAuthorizedRadio(!!legalYes);
      assertYesNoByQuestion(/are\s+you\s+legally\s+authorized[\s\S]*united\s*states/i, !!legalYes);
    }
    // Sponsorship
    const visaPref = (profile.visaSponsorship || '').toLowerCase();
    const sponsorYes = (/^\s*yes\b/i.test(visaPref)) ||
                       ((/\brequire\b/i.test(visaPref)) && !(/\bdo\s*not\s*require\b|\bdo\s*not\b|\bno\b/i.test(visaPref)));
    clickSponsorshipRadio(!!sponsorYes);
    // Stabilize sponsorship answer across re-renders and select dropdowns
    assertYesNoByQuestion(/(require\s+sponsorship|sponsorship\s+for\s+employment|visa\s*status|h-?1b)/i, !!sponsorYes);
    // NYC office generic + specific phrasing
    clickRadioByQuestion(/(nyc|new\s+york|hybrid)/i, !!nycWantYes);
    clickNycTueThuRadio(!!nycWantYes);
    // Relocate to NYC question (if shown on some forms)
    clickRelocateNYCRadio(!!nycWantYes);
    assertYesNoByQuestion(/(located[\s\S]*relocate[\s\S]*nyc|willing[\s\S]*relocate[\s\S]*nyc|relocate\s*to\s*nyc)/i, !!nycWantYes);
    // After clicking radios, restore any text inputs that might have been cleared by re-render
    setTimeout(() => restoreTextInputs(snapshot), 50);
    
    // Always answer "No" to transgender identity question
    try {
      const transRx = /(do\s*you\s*identify\s*as\s*transgender|identify\s*as\s*transgender|are\s*you\s*transgender)/i;
      clickRadioByQuestion(transRx, false);
      assertYesNoByQuestion(transRx, false);
    } catch(_) {}

    // Fill remaining common yes/no groups with sensible defaults
    try { fillAllYesNoDefaults(); } catch(_) {}
    
    // AUTO FILL RESUME
    console.log('🔍 CHECKING RESUME SECTION...', {
      hasResumeFileName: !!profile.resumeFileName,
      resumeFileName: profile.resumeFileName || 'none'
    });
    
    if (profile.resumeFileName) {
      console.log('🚀 AUTO FILLING RESUME...');
      console.log('Resume data available:', {
        fileName: profile.resumeFileName,
        hasFileData: !!profile.resumeFileData,
        fileDataLength: profile.resumeFileData ? profile.resumeFileData.length : 0
      });
      
      // Find and auto-fill resume upload field
      const allFileInputs = document.querySelectorAll('input[type="file"]');
      console.log(`Found ${allFileInputs.length} file input fields on the page`);
      
      let resumeFieldFilled = false;
      
      for (const fileInput of allFileInputs) {
        // Check if it's likely a resume field
        const parent = fileInput.closest('div, fieldset, form, section');
        const parentText = parent ? parent.textContent.toLowerCase() : '';
        const name = (fileInput.name || '').toLowerCase();
        const id = (fileInput.id || '').toLowerCase();
        const placeholder = (fileInput.placeholder || '').toLowerCase();
        
        const isResumeField = parentText.includes('resume') || parentText.includes('cv') || 
                             parentText.includes('document') || name.includes('resume') || 
                             name.includes('cv') || id.includes('resume') || id.includes('cv') ||
                             placeholder.includes('resume') || placeholder.includes('cv');
        
        if (isResumeField || allFileInputs.length === 1) {
          console.log('🎯 Found resume upload field - attempting auto-fill');
          
          // Prepare resume data for upload
          const resumeData = {
            fileName: profile.resumeFileName,
            file: profile.resumeFileData
          };
          
          // Try to auto-fill the resume
          if (fillResumeField(fileInput, resumeData)) {
            resumeFieldFilled = true;
            console.log(`✅ Auto-filled resume field`);
            filledCount++;
            break;
          } else {
            console.log(`❌ Auto-fill failed, trying alternative method`);
            
            // Alternative: Try to trigger file input click
            try {
              fileInput.click();
              console.log(`✅ Triggered file input click`);
              resumeFieldFilled = true;
              filledCount++;
              break;
            } catch (error) {
              console.log(`❌ File input click failed:`, error);
            }
          }
        }
      }
      
      if (resumeFieldFilled) {
        showNotification(`📄 Resume auto-filled: ${profile.resumeFileName}`, 'success');
      } else {
        console.log('⚠️ No resume upload field found on this page');
        showNotification(`📄 No resume upload field found. Your saved resume: ${profile.resumeFileName}`, 'warning');
      }
    }
    
    if (profile.visaSponsorship) {
      console.log('🚀 ULTRA AGGRESSIVE VISA SPONSORSHIP FILL STARTING...');
      if (fillCheckboxRadio('visa', profile.visaSponsorship)) {
        filledCount++;
        console.log(`✅ ULTRA AGGRESSIVE filled visa sponsorship`);
      }
      // Also use the robust radio selector (same pattern as NYC)
      const visaWantYes = /^(yes|i\s+will\s+require|require)/i.test((profile.visaSponsorship||''));
      clickRadioByQuestion(/(visa|sponsorship|employment\s+visa\s+status|immigration)/i, !!visaWantYes);
    }
    
    console.log(`=== FILL COMPLETE: ${filledCount} fields filled ===`);
    
    // Show notification
    if (filledCount > 0) {
      showNotification(`🎉 SUCCESS! Filled ${filledCount} fields!`, 'success');
    } else {
      showNotification(`❌ No fields found to fill. Check console for details.`, 'error');
    }
    
  } catch (error) {
    console.error('Error in fillNow:', error);
    showNotification(`❌ Error: ${error.message}`, 'error');
  }
}

// Show notification
function showNotification(message, type = 'success') {
  const notification = document.createElement('div');
  notification.style.cssText = `
    position: fixed; top: 20px; right: 20px; color: white;
    padding: 15px 25px; border-radius: 8px; z-index: 10000; 
    font-family: Arial, sans-serif; box-shadow: 0 4px 20px rgba(0,0,0,0.3); 
    font-size: 14px; font-weight: bold; max-width: 300px;
    background: ${type === 'success' ? '#4CAF50' : '#f44336'};
  `;
  notification.textContent = message;
  document.body.appendChild(notification);
  setTimeout(() => notification.remove(), 5000);
}

// Listen for the fill event
window.addEventListener('JOB_AUTOFILL_NOW', () => {
  console.log('Fill event received - starting ultra aggressive fill');
  fillNow();
});

// Also listen for messages from popup
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'FILL_FORM') {
    fillNow();
    sendResponse({success: true});
  }
});

// On-demand resume upload using resume stored in the widget (chrome.storage.local)
async function uploadResumeFromStorage() {
  try {
    const { jobProfile } = await chrome.storage.local.get(['jobProfile']);
    if (!jobProfile?.resumeFileData || !jobProfile?.resumeFileName) {
      console.log('RESUME> No resume saved in widget storage.');
      return false;
    }
    const resumeData = { fileName: jobProfile.resumeFileName, file: jobProfile.resumeFileData };

    // Try native inputs first
    const inputs = document.querySelectorAll('input[type="file"]');
    for (const input of inputs) {
      const ok = await fillResumeField(input, resumeData);
      if (ok) { console.log('RESUME> ✅ Uploaded via file input'); return true; }
    }

    // Try common dropzones
    const candidates = Array.from(document.querySelectorAll('div,section,form'))
      .filter(el => /drag\s+and\s+drop|browse\s+computer|upload/i.test(el.textContent||''));
    if (candidates.length) {
      const dropEl = candidates.sort((a,b)=>{
        const ra=a.getBoundingClientRect?.()||{width:0,height:0};
        const rb=b.getBoundingClientRect?.()||{width:0,height:0};
        return (rb.width*rb.height)-(ra.width*ra.height);
      })[0];

      // Build File
      const base64 = resumeData.file.split(',')[1];
      const bytes = atob(base64);
      const nums = new Array(bytes.length);
      for (let i=0;i<bytes.length;i++) nums[i] = bytes.charCodeAt(i);
      const arr = new Uint8Array(nums);
      const blob = new Blob([arr], { type: getMimeType(resumeData.fileName) });
      const file = new File([blob], resumeData.fileName, { type: blob.type });
      const dt = new DataTransfer(); dt.items.add(file);

      ['dragenter','dragover','drop'].forEach(t => dropEl.dispatchEvent(new DragEvent(t, { bubbles:true, dataTransfer: dt })));
      console.log('RESUME> ✅ Uploaded via dropzone');
      return true;
    }

    console.log('RESUME> ❌ No file input or dropzone found');
    return false;
  } catch (e) {
    console.log('RESUME> Error:', e);
    return false;
  }
}

// Expose a window event so users can trigger upload from page console (no chrome.* there)
window.addEventListener('JOB_AUTOFILL_UPLOAD_RESUME', () => {
  console.log('RESUME> Upload request received');
  uploadResumeFromStorage();
});

// Function to handle resume file upload
async function fillResumeField(fileInput, resumeData) {
  console.log('fillResumeField called with:', {
    hasResumeData: !!resumeData,
    hasFile: !!(resumeData && resumeData.file),
    hasFileName: !!(resumeData && resumeData.fileName),
    fileName: resumeData ? resumeData.fileName : 'none'
  });
  
  if (!resumeData || !resumeData.file || !resumeData.fileName) {
    console.log('No resume data provided - missing:', {
      resumeData: !!resumeData,
      file: !!(resumeData && resumeData.file),
      fileName: !!(resumeData && resumeData.fileName)
    });
    return false;
  }
  
  console.log(`Attempting to upload resume: ${resumeData.fileName}`);
  
  try {
    // Convert base64 back to file
    const base64Data = resumeData.file.split(',')[1];
    const byteCharacters = atob(base64Data);
    const byteNumbers = new Array(byteCharacters.length);
    for (let i = 0; i < byteCharacters.length; i++) byteNumbers[i] = byteCharacters.charCodeAt(i);
    const byteArray = new Uint8Array(byteNumbers);
    const blob = new Blob([byteArray], { type: getMimeType(resumeData.fileName) });
    const file = new File([blob], resumeData.fileName, { type: blob.type });
    
    // Try multiple aggressive methods to set the file
    
    // Method 1: DataTransfer
    console.log('Trying method 1: DataTransfer');
    try {
      const dataTransfer = new DataTransfer();
      dataTransfer.items.add(file);
      fileInput.files = dataTransfer.files;
      
      // Trigger comprehensive events
      const events = ['focus', 'input', 'change', 'blur', 'fileSelected', 'upload', 'drop'];
      events.forEach(eventType => {
        fileInput.dispatchEvent(new Event(eventType, { bubbles: true }));
      });
      
      console.log('DataTransfer method completed');
    } catch (e) {
      console.log('DataTransfer failed:', e);
    }
    
    // Method 2: Direct property assignment with Object.defineProperty
    console.log('Trying method 2: Direct assignment');
    try {
      Object.defineProperty(fileInput, 'files', {
        value: [file],
        writable: true,
        configurable: true
      });
      
      // Trigger events again
      fileInput.dispatchEvent(new Event('change', { bubbles: true }));
      fileInput.dispatchEvent(new Event('input', { bubbles: true }));
      
      console.log('Direct assignment completed');
    } catch (e) {
      console.log('Direct assignment failed:', e);
    }
    
    // Method 3: Try setting value property
    console.log('Trying method 3: Value property');
    try {
      fileInput.value = resumeData.fileName;
      fileInput.dispatchEvent(new Event('change', { bubbles: true }));
      fileInput.dispatchEvent(new Event('input', { bubbles: true }));
      console.log('Value property completed');
    } catch (e) {
      console.log('Value property failed:', e);
    }
    
    // Visual feedback
    fileInput.style.outline = '3px solid #4CAF50';
    fileInput.style.backgroundColor = '#e8f5e8';
    
    // Force a re-render by temporarily hiding and showing
    const originalDisplay = fileInput.style.display;
    fileInput.style.display = 'none';
    setTimeout(() => {
      fileInput.style.display = originalDisplay;
    }, 10);
    
    console.log(`✅ Resume uploaded successfully: ${resumeData.fileName}`);
    
    // Show success notification
    showNotification(`📄 Resume uploaded: ${resumeData.fileName}`, 'success');
    
    return true;
    
  } catch (error) {
    console.error('Error uploading resume:', error);
    showNotification('❌ Resume upload failed - trying alternative method', 'error');

    // Fallback 1: try dropzone (drag & drop)
    try {
      const dropCandidates = Array.from(document.querySelectorAll('div,section,form'))
        .filter(el => (/drag\s+and\s+drop|drop\s+files|browse\s+computer/i).test((el.textContent||'')));
      const dropEl = dropCandidates.sort((a,b)=>((b.getBoundingClientRect?.().width||0)*(b.getBoundingClientRect?.().height||0))-
                                           ((a.getBoundingClientRect?.().width||0)*(a.getBoundingClientRect?.().height||0)))[0];
      if (dropEl) {
        const dt = new DataTransfer();
        // Recreate File from stored base64
        const base64 = resumeData.file.split(',')[1];
        const bytes = atob(base64);
        const nums = new Array(bytes.length);
        for (let i=0;i<bytes.length;i++) nums[i] = bytes.charCodeAt(i);
        const arr = new Uint8Array(nums);
        const blob = new Blob([arr], { type: getMimeType(resumeData.fileName) });
        const file = new File([blob], resumeData.fileName, { type: blob.type });
        dt.items.add(file);
        const events = ['dragenter','dragover','drop'];
        events.forEach(type => dropEl.dispatchEvent(new DragEvent(type, { bubbles: true, dataTransfer: dt })));
        console.log('✅ Dropzone upload attempted');
        return true;
      }
    } catch (e) {
      console.log('Dropzone fallback failed:', e);
    }

    // Fallback 2: trigger the file input dialog (user selects manually)
    try { fileInput.click(); } catch(_e) {}
    return false;
  }
}

// Helper function to get MIME type from file extension
function getMimeType(fileName) {
  const ext = fileName.split('.').pop().toLowerCase();
  const mimeTypes = {
    'pdf': 'application/pdf',
    'doc': 'application/msword',
    'docx': 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
    'txt': 'text/plain'
  };
  return mimeTypes[ext] || 'application/octet-stream';
}

// Normalize GitHub input to a URL the page will accept
function normalizeGithubUrl(value) {
  try {
    if (!value) return '';
    let v = String(value).trim();
    if (/^https?:\/\//i.test(v)) return v;
    if (/^www\./i.test(v)) return `https://${v}`;
    if (/^github\.com\//i.test(v)) return `https://${v}`;
    if (/^[A-Za-z0-9-]+$/i.test(v)) return `https://github.com/${v}`;
    return `https://${v}`;
  } catch(_) { return value; }
}

// Alternative approach: Auto-fill resume by triggering upload dialog
function triggerResumeUpload() {
  console.log('🚀 Looking for resume upload fields...');
  
  const resumeSelectors = [
    'input[type="file"][name*="resume"]',
    'input[type="file"][name*="cv"]',
    'input[type="file"][name*="document"]',
    'input[type="file"][id*="resume"]',
    'input[type="file"][id*="cv"]',
    'input[type="file"]'
  ];
  
  for (const selector of resumeSelectors) {
    const fileInput = document.querySelector(selector);
    if (fileInput) {
      console.log('Found resume upload field:', fileInput);
      
      // Check if it's likely a resume field by looking at surrounding text
      const parent = fileInput.closest('div, fieldset, form, section');
      const parentText = parent ? parent.textContent.toLowerCase() : '';
      
      if (parentText.includes('resume') || parentText.includes('cv') || parentText.includes('document')) {
        console.log('🎯 Confirmed resume field - triggering upload dialog');
        
        // Trigger the file input
        fileInput.click();
        
        // Visual feedback
        fileInput.style.outline = '3px solid #4CAF50';
        fileInput.style.backgroundColor = '#e8f5e8';
        
        // Show notification
        showNotification('📄 Resume upload dialog opened - Please select your resume file', 'info');
        
        return true;
      }
    }
  }
  
  console.log('No resume upload fields found');
  return false;
}

// Debug function to check resume upload status
function debugResumeUpload() {
  console.log('=== RESUME UPLOAD DEBUG ===');
  
  const allFileInputs = document.querySelectorAll('input[type="file"]');
  console.log(`Found ${allFileInputs.length} file input fields:`);
  
  allFileInputs.forEach((input, index) => {
    console.log(`File Input ${index + 1}:`);
    console.log(`  - Element:`, input);
    console.log(`  - Name: "${input.name}"`);
    console.log(`  - ID: "${input.id}"`);
    console.log(`  - Class: "${input.className}"`);
    console.log(`  - Files: ${input.files.length} files selected`);
    
    if (input.files.length > 0) {
      console.log(`  - File name: ${input.files[0].name}`);
      console.log(`  - File size: ${input.files[0].size} bytes`);
      console.log(`  - File type: ${input.files[0].type}`);
    }
    
    // Check surrounding elements
    const parent = input.closest('div, fieldset, form, section');
    if (parent) {
      console.log(`  - Parent text: "${parent.textContent.substring(0, 100)}..."`);
    }
  });
  
  // Check if any file inputs have files
  const inputsWithFiles = Array.from(allFileInputs).filter(input => input.files.length > 0);
  console.log(`File inputs with files: ${inputsWithFiles.length}`);
  
  return allFileInputs.length;
}



// Debug function to analyze all form fields on the page
function debugAllFields() {
  console.log('=== FORM FIELD ANALYSIS ===');
  
  const allElements = document.querySelectorAll('input, textarea, select, button');
  console.log(`Found ${allElements.length} total form elements:`);
  
  allElements.forEach((element, index) => {
    console.log(`Element ${index + 1}:`, {
      tagName: element.tagName,
      type: element.type || 'N/A',
      name: element.name || 'N/A',
      id: element.id || 'N/A',
      placeholder: element.placeholder || 'N/A',
      className: element.className || 'N/A',
      value: element.value || 'N/A',
      parentText: element.parentElement ? element.parentElement.textContent.substring(0, 100) : 'N/A'
    });
  });
  
  // Check for specific field types
  const textInputs = document.querySelectorAll('input[type="text"]');
  const emailInputs = document.querySelectorAll('input[type="email"]');
  const phoneInputs = document.querySelectorAll('input[type="tel"]');
  const selects = document.querySelectorAll('select');
  const textareas = document.querySelectorAll('textarea');
  
  console.log(`Field breakdown:`);
  console.log(`- Text inputs: ${textInputs.length}`);
  console.log(`- Email inputs: ${emailInputs.length}`);
  console.log(`- Phone inputs: ${phoneInputs.length}`);
  console.log(`- Select dropdowns: ${selects.length}`);
  console.log(`- Textareas: ${textareas.length}`);
  
  return allElements.length;
}

// Make debug functions available globally
window.debugResumeUpload = debugResumeUpload;
window.debugAllFields = debugAllFields;

console.log('Ultra aggressive content script ready - WILL find and fill phone fields AND resume uploads!');
console.log('Debug: Run debugResumeUpload() in console to check resume upload status');
console.log('Debug: Run debugAllFields() in console to analyze all form fields on the page');

// ===== Gender Autofill (radios + optional self-describe) =====
// Handles: "How would you describe your gender?"
// Uses widget storage: jobProfile.gender, jobProfile.genderSelfDescribe
(function setupGenderAutofill() {
  function normalizeText(s) {
    return (s || '').toLowerCase().replace(/[^\p{L}\p{N}]+/gu, ' ').trim();
  }

  function isGenderDescribeQuestion(text) {
    const t = normalizeText(text);
    return (
      t.includes('how would you describe your gender') ||
      t.includes('self describe gender') ||
      t.includes('please describe your gender') ||
      t.includes('gender self describe') ||
      t.includes('self describe') && t.includes('gender')
    );
  }

  function setNative(input, value) {
    try {
      const proto = Object.getPrototypeOf(input) || HTMLInputElement.prototype;
      const setter = Object.getOwnPropertyDescriptor(proto, 'value')?.set || Object.getOwnPropertyDescriptor(HTMLInputElement.prototype, 'value')?.set;
      if (setter) setter.call(input, value); else input.value = value;
    } catch(_) { input.value = value; }
    input.dispatchEvent(new Event('input', { bubbles: true }));
    input.dispatchEvent(new Event('change', { bubbles: true }));
  }

  function findQuestionRoot() {
    const anchors = Array.from(document.querySelectorAll('label,legend,h1,h2,h3,h4,h5,h6,p,div,section'));
    const a = anchors.find(el => isGenderDescribeQuestion(el.textContent || ''));
    if (!a) return null;
    return a.closest('fieldset,section,form,div') || a.parentElement || a;
  }

  function textOf(el) {
    const aria = el.getAttribute && (el.getAttribute('aria-label') || '');
    return normalizeText(`${aria} ${el.textContent || ''}`);
  }

  function labelForInput(input) {
    if (!input) return '';
    if (input.id) {
      const l = document.querySelector(`label[for="${input.id}"]`);
      if (l) return l.textContent || '';
    }
    const wrap = input.closest('label');
    return wrap ? (wrap.textContent || '') : '';
  }

  function clickEl(el) {
    if (!el) return false;
    el.scrollIntoView?.({ block: 'center' });
    ['pointerdown','mousedown','mouseup','click'].forEach(t => el.dispatchEvent(new MouseEvent(t, { bubbles: true })));
    if (el.type === 'radio') {
      el.checked = true;
      el.dispatchEvent(new Event('input', { bubbles: true }));
      el.dispatchEvent(new Event('change', { bubbles: true }));
    }
    return true;
  }

  function selectGenderOption(root, desired) {
    const d = normalizeText(desired);
    const isMan = /\bman\b/.test(d);
    const isWoman = /\bwoman\b/.test(d);
    const isNb = /(non\s*binary|nonbinary)/.test(d);
    const isPreferNot = /(prefer\s*not\s*to\s*answer|do\s*not\s*wish\s*to\s*answer|prefer\s*not\s*to\s*say)/.test(d);
    const isSelf = /(self\s*describe)/.test(d);

    const tokens = [
      { on: isMan, rx: /\bman\b/ },
      { on: isWoman, rx: /\bwoman\b/ },
      { on: isNb, rx: /(non\s*binary|nonbinary)/ },
      { on: isSelf, rx: /(self\s*describe)/ },
      { on: isPreferNot, rx: /(wish\s*to\s*answer|prefer\s*not)/ }
    ];

    const radios = Array.from(root.querySelectorAll('input[type="radio"], [role="radio"], label'));
    for (const { on, rx } of tokens) {
      if (!on) continue;
      let target = radios.find(e => rx.test(textOf(e)));
      if (!target) {
        const inputs = Array.from(root.querySelectorAll('input[type="radio"]'));
        const match = inputs.find(r => rx.test(normalizeText(`${r.value || ''} ${labelForInput(r)}`)));
        if (match) target = match;
      }
      if (target) return clickEl(target.closest('[role="radio"],label') || target);
    }
    return false;
  }

  async function fillGenderFromStorage() {
    try {
      if (typeof chrome === 'undefined' || !chrome.storage?.local) return false;
      const { jobProfile } = await chrome.storage.local.get(['jobProfile']);
      if (!jobProfile) return false;
      const root = findQuestionRoot();
      if (!root) return false;

      const gender = (jobProfile.gender || '').trim();
      const selfDesc = (jobProfile.genderSelfDescribe || '').trim();

      if (selfDesc) {
        // Select self-describe then wait for input to appear
        selectGenderOption(root, 'self describe');
        let input = null; let attempts = 0;
        while (!input && attempts < 10) {
          input = root.querySelector('input[type="text"],textarea,input:not([type])');
          if (!input) await new Promise(r => setTimeout(r, 120));
          attempts++;
        }
        if (input) {
          setNative(input, selfDesc);
          input.style.outline = '3px solid #4CAF50';
          input.style.backgroundColor = '#e8f5e8';
          return true;
        }
        return false;
      }

      if (gender) {
        return selectGenderOption(root, gender);
      }
      return false;
    } catch(_) { return false; }
  }

  // Console-triggerable event
  window.addEventListener('JOB_AUTOFILL_GENDER_SELF_DESCRIBE', () => {
    fillGenderFromStorage().then(ok => console.log('GENDER> filled:', ok));
  });
  window.addEventListener('JOB_AUTOFILL_GENDER', () => {
    fillGenderFromStorage().then(ok => console.log('GENDER> filled:', ok));
  });

  // Re-assert on DOM mutations if the question exists
  const mo = new MutationObserver(() => {
    const root = findQuestionRoot();
    if (root && root.dataset.genderFilledOnce !== '1') {
      fillGenderFromStorage().then(ok => {
        if (ok && root) root.dataset.genderFilledOnce = '1';
      });
    }
  });
  mo.observe(document.documentElement, { childList: true, subtree: true });
})();

// Click all common Yes/No radio groups with sensible defaults
function fillAllYesNoDefaults() {
  try {
    const yesDefaults = [
      /legally\s+authorized[\s\S]*united\s*states/i, // work authorization
      /work\s+in\s+the\s+united\s+states/i,
      /located\s+in[,\s]*or\s+willing\s+to\s+relocate/i,
      /work\s+from\s+our\s+nyc\s+office/i
    ];
    const noDefaults = [
      /(require|need)[\s\S]*sponsorship/i, // visa sponsorship
      /(identify\s*as\s*transgender)/i
    ];

    yesDefaults.forEach(rx => { try { clickRadioByQuestion(rx, true); assertYesNoByQuestion(rx, true); } catch(_) {} });
    noDefaults.forEach(rx => { try { clickRadioByQuestion(rx, false); assertYesNoByQuestion(rx, false); } catch(_) {} });

    // Fallback: generic containers with two options Yes/No
    const containers = Array.from(document.querySelectorAll('fieldset,section,form,div'))
      .filter(n => /(\byes\b).*(\bno\b)|(\bno\b).*(\byes\b)/i.test(n.textContent || ''))
      .slice(0, 20);
    containers.forEach(root => {
      const picked = root.querySelector('[aria-pressed="true"], input[type="radio"]:checked');
      if (picked) return;
      const yesRx = /\byes\b/i; const noRx = /\bno\b/i;
      let yesEl = Array.from(root.querySelectorAll('label,[role="radio"],button,div,span')).find(e => yesRx.test((e.getAttribute?.('aria-label')||'') + ' ' + (e.textContent||'')));
      let noEl = Array.from(root.querySelectorAll('label,[role="radio"],button,div,span')).find(e => noRx.test((e.getAttribute?.('aria-label')||'') + ' ' + (e.textContent||'')));
      // Default policy: prefer Yes unless text hints sponsorship/visa
      const txt = (root.textContent||'').toLowerCase();
      const wantYes = !(/sponsor|sponsorship|immigration|visa/.test(txt));
      const target = wantYes ? (yesEl || root.querySelector('input[type="radio"]')) : (noEl || root.querySelector('input[type="radio"]'));
      if (target) {
        ['pointerdown','mousedown','mouseup','click'].forEach(t => target.dispatchEvent(new MouseEvent(t, { bubbles: true })));
      }
    });
  } catch(_) {}
}

// Console event to fill all radios at once
window.addEventListener('JOB_AUTOFILL_ALL_YES_NO', () => { try { fillAllYesNoDefaults(); fillAllRadiosOnceFromProfile(); } catch(_) {} });

// Stronger pass: fill every Yes/No radio group once using widget values and rules
async function fillAllRadiosOnceFromProfile() {
  try {
    if (typeof chrome === 'undefined' || !chrome.storage?.local) return false;
    const { jobProfile } = await chrome.storage.local.get(['jobProfile']);
    const profile = jobProfile || {};

    const allContainers = Array.from(document.querySelectorAll('fieldset,section,form,div'));
    const groups = allContainers
      .filter(n => {
        if (!n.querySelector('input[type="radio"],[role="radio"],button')) return false;
        const t = (n.textContent || '');
        const hasTextYes = /\byes\b/i.test(t);
        const hasTextNo  = /\bno\b/i.test(t);
        if (hasTextYes && hasTextNo) return true;
        const cand = Array.from(n.querySelectorAll('[aria-label],[role="radio"],button,div,span,label'));
        const hasAriaYes = cand.some(e => /\byes\b/i.test(((e.getAttribute?.('aria-label')||'') + ' ' + (e.textContent||''))));
        const hasAriaNo  = cand.some(e => /\bno\b/i.test(((e.getAttribute?.('aria-label')||'') + ' ' + (e.textContent||''))));
        return hasAriaYes && hasAriaNo;
      })
      .sort((a,b)=> (a.textContent||'').length - (b.textContent||'').length);

    const rx = {
      legal: /are\s+you\s+legally\s+authorized[\s\S]*united\s*states/i,
      sponsor: /(require|need)[\s\S]*sponsorship|h-?1b|visa\s*status/i,
      nyc: /(nyc|new\s+york|work\s+from\s+our\s+nyc|hybrid)/i,
      relocateNyc: /(located[\s\S]*relocate[\s\S]*nyc|willing[\s\S]*relocate[\s\S]*nyc|relocate\s*to\s*nyc)/i,
      transgender: /(do\s*you\s*identify\s*as\s*transgender)/i
    };

    const decide = (text) => {
      const t = (text||'').toLowerCase();
      if (rx.transgender.test(t)) return false; // always No
      if (rx.legal.test(t)) {
        if (!profile.workAuth) return null; // skip if not provided
        const w = (profile.workAuth||'').toLowerCase();
        return (w.includes('legal') || /^\s*yes\b/.test(w));
      }
      if (rx.sponsor.test(t)) {
        const v = (profile.visaSponsorship||'').toLowerCase();
        if (v) return (/^\s*yes\b/.test(v) || /require/.test(v));
        return false; // default No
      }
      if (rx.nyc.test(t) || rx.relocateNyc.test(t)) {
        const n = (profile.nycOffice||'').toLowerCase();
        if (n) return /^\s*yes\b/.test(n);
        return true; // lean Yes
      }
      return true; // generic default Yes
    };

    const clickWithin = (root, wantYes) => {
      try {
        const token = wantYes ? 'yes' : 'no';
        const tokenRx = new RegExp(`(^|\\s)${token}(\\s|$)`, 'i');
        let target = Array.from(root.querySelectorAll('label,[role="radio"],button,div,span')).find(e => tokenRx.test(((e.getAttribute?.('aria-label')||'') + ' ' + (e.textContent||''))));
        if (!target) {
          const yesNoValRx = wantYes ? /(yes|true)/i : /(no|false)/i;
          const r = Array.from(root.querySelectorAll('input[type="radio"]')).find(input => yesNoValRx.test(input.value||'') || tokenRx.test((document.querySelector(`label[for="${input.id}"]`)?.textContent||'')));
          if (r) target = r;
        }
        if (!target) return false;
        ['pointerdown','mousedown','mouseup','click'].forEach(t => target.dispatchEvent(new MouseEvent(t, { bubbles: true })));
        if (target.type === 'radio') { target.checked = true; target.dispatchEvent(new Event('input',{bubbles:true})); target.dispatchEvent(new Event('change',{bubbles:true})); }
        return true;
      } catch(_) { return false; }
    };

    for (const g of groups) {
      if (g.querySelector('[aria-pressed="true"], input[type="radio"]:checked')) continue; // already answered
      const want = decide(g.textContent||'');
      if (want === null) continue; // intentionally skipped (e.g., legal w/o data)
      clickWithin(g, !!want);
    }
    return true;
  } catch(_) { return false; }
}

// Slower, staggered fill with retries per group (for SPA re-renders)
async function fillAllRadiosStaggeredFromProfile(delayMs = 450) {
  try {
    if (typeof chrome === 'undefined' || !chrome.storage?.local) return false;
    const { jobProfile } = await chrome.storage.local.get(['jobProfile']);
    const profile = jobProfile || {};

    const allContainers = Array.from(document.querySelectorAll('fieldset,section,form,div'));
    const groups = allContainers
      .filter(n => {
        if (!n.querySelector('input[type="radio"],[role="radio"],button')) return false;
        const t = (n.textContent || '');
        const hasTextYes = /\byes\b/i.test(t);
        const hasTextNo  = /\bno\b/i.test(t);
        if (hasTextYes && hasTextNo) return true;
        const cand = Array.from(n.querySelectorAll('[aria-label],[role="radio"],button,div,span,label'));
        const hasAriaYes = cand.some(e => /\byes\b/i.test(((e.getAttribute?.('aria-label')||'') + ' ' + (e.textContent||''))));
        const hasAriaNo  = cand.some(e => /\bno\b/i.test(((e.getAttribute?.('aria-label')||'') + ' ' + (e.textContent||''))));
        return hasAriaYes && hasAriaNo;
      })
      .sort((a,b)=> (a.textContent||'').length - (b.textContent||'').length);

    const rx = {
      legal: /are\s+you\s+legally\s+authorized[\s\S]*united\s*states/i,
      sponsor: /(require|need)[\s\S]*sponsorship|h-?1b|visa\s*status/i,
      nyc: /(nyc|new\s+york|work\s+from\s+our\s+nyc|hybrid)/i,
      relocateNyc: /(located[\s\S]*relocate[\s\S]*nyc|willing[\s\S]*relocate[\s\S]*nyc|relocate\s*to\s*nyc)/i,
      transgender: /(do\s*you\s*identify\s*as\s*transgender)/i
    };

    const decide = (text) => {
      const t = (text||'').toLowerCase();
      if (rx.transgender.test(t)) return false; // always No
      if (rx.legal.test(t)) {
        if (!profile.workAuth) return null; // skip if not provided
        const w = (profile.workAuth||'').toLowerCase();
        return (w.includes('legal') || /^\s*yes\b/.test(w));
      }
      if (rx.sponsor.test(t)) {
        const v = (profile.visaSponsorship||'').toLowerCase();
        if (v) return (/^\s*yes\b/.test(v) || /require/.test(v));
        return false; // default No
      }
      if (rx.nyc.test(t) || rx.relocateNyc.test(t)) {
        const n = (profile.nycOffice||'').toLowerCase();
        if (n) return /^\s*yes\b/.test(n);
        return true; // lean Yes
      }
      return true; // generic default Yes
    };

    const clickWithin = (root, wantYes) => {
      try {
        const token = wantYes ? 'yes' : 'no';
        const tokenRx = new RegExp(`(^|\\s)${token}(\\s|$)`, 'i');
        let target = Array.from(root.querySelectorAll('label,[role="radio"],button,div,span')).find(e => tokenRx.test(((e.getAttribute?.('aria-label')||'') + ' ' + (e.textContent||''))));
        if (!target) {
          const yesNoValRx = wantYes ? /(yes|true)/i : /(no|false)/i;
          const r = Array.from(root.querySelectorAll('input[type="radio"]')).find(input => yesNoValRx.test(input.value||'') || tokenRx.test((document.querySelector(`label[for="${input.id}"]`)?.textContent||'')));
          if (r) target = r;
        }
        if (!target) return false;
        try { (target.closest('[role="radio"],button,label,div,span') || target).scrollIntoView({ block: 'center' }); } catch(_) {}
        ['pointerdown','mousedown','mouseup','click'].forEach(t => target.dispatchEvent(new MouseEvent(t, { bubbles: true })));
        if (target.type === 'radio') { target.checked = true; target.dispatchEvent(new Event('input',{bubbles:true})); target.dispatchEvent(new Event('change',{bubbles:true})); }
        return true;
      } catch(_) { return false; }
    };

    for (const g of groups) {
      if (g.querySelector('[aria-pressed="true"], input[type="radio"]:checked')) continue;
      const want = decide(g.textContent||'');
      if (want === null) { await new Promise(r => setTimeout(r, delayMs)); continue; }
      let ok = false; let attempts = 0;
      while (!ok && attempts < 5) {
        ok = clickWithin(g, !!want);
        if (!ok) await new Promise(r => setTimeout(r, 300));
        attempts++;
      }
      await new Promise(r => setTimeout(r, delayMs));
    }
    // Re-assert once more after a short global delay to survive re-renders
    await new Promise(r => setTimeout(r, Math.max(600, delayMs)));
    try { await fillAllRadiosOnceFromProfile(); } catch(_) {}
    return true;
  } catch(_) { return false; }
}

// Console event: slower paced all-radios fill
window.addEventListener('JOB_AUTOFILL_ALL_YES_NO_SLOW', () => { fillAllRadiosStaggeredFromProfile(500); });
// Even slower variant
window.addEventListener('JOB_AUTOFILL_ALL_YES_NO_SLOWER', () => { fillAllRadiosStaggeredFromProfile(1200); });

// Persistently retry until all visible Yes/No groups are filled or max rounds reached
async function fillAllRadiosUntilComplete(maxRounds = 10, groupDelay = 450, roundDelay = 900) {
  try {
    if (typeof chrome === 'undefined' || !chrome.storage?.local) return false;
    const { jobProfile } = await chrome.storage.local.get(['jobProfile']);
    const profile = jobProfile || {};

    const getGroups = () => {
      const allContainers = Array.from(document.querySelectorAll('fieldset,section,form,div'));
      return allContainers
        .filter(n => {
          if (!n.querySelector('input[type="radio"],[role="radio"],button')) return false;
          const t = (n.textContent || '');
          const hasTextYes = /\byes\b/i.test(t);
          const hasTextNo  = /\bno\b/i.test(t);
          if (hasTextYes && hasTextNo) return true;
          const cand = Array.from(n.querySelectorAll('[aria-label],[role="radio"],button,div,span,label'));
          const hasAriaYes = cand.some(e => /\byes\b/i.test(((e.getAttribute?.('aria-label')||'') + ' ' + (e.textContent||''))));
          const hasAriaNo  = cand.some(e => /\bno\b/i.test(((e.getAttribute?.('aria-label')||'') + ' ' + (e.textContent||''))));
          return hasAriaYes && hasAriaNo;
        })
        .sort((a,b)=> (a.textContent||'').length - (b.textContent||'').length);
    };

    const isAnswered = (root) => {
      if (root.querySelector('input[type="radio"]:checked')) return true;
      if (root.querySelector('[aria-checked="true"], [aria-pressed="true"]')) return true;
      const roles = Array.from(root.querySelectorAll('[role="radio"]'));
      if (roles.some(el => (el.getAttribute('aria-checked') === 'true'))) return true;
      return false;
    };

    const rx = {
      legal: /are\s+you\s+legally\s+authorized[\s\S]*united\s*states/i,
      sponsor: /(require|need)[\s\S]*sponsorship|h-?1b|visa\s*status/i,
      nyc: /(nyc|new\s+york|work\s+from\s+our\s+nyc|hybrid)/i,
      relocateNyc: /(located[\s\S]*relocate[\s\S]*nyc|willing[\s\S]*relocate[\s\S]*nyc|relocate\s*to\s*nyc)/i,
      transgender: /(do\s*you\s*identify\s*as\s*transgender)/i
    };

    const decide = (text) => {
      const t = (text||'').toLowerCase();
      if (rx.transgender.test(t)) return false; // always No
      if (rx.legal.test(t)) {
        if (!profile.workAuth) return null; // skip if not provided
        const w = (profile.workAuth||'').toLowerCase();
        return (w.includes('legal') || /^\s*yes\b/.test(w));
      }
      if (rx.sponsor.test(t)) {
        const v = (profile.visaSponsorship||'').toLowerCase();
        if (v) return (/^\s*yes\b/.test(v) || /require/.test(v));
        return false; // default No
      }
      if (rx.nyc.test(t) || rx.relocateNyc.test(t)) {
        const n = (profile.nycOffice||'').toLowerCase();
        if (n) return /^\s*yes\b/.test(n);
        return true; // lean Yes
      }
      return true; // generic default Yes
    };

    function findYesNoInRoot(root) {
      const tokenRx = {
        yes: /(^|\s)yes(\s|$)/i,
        no: /(^|\s)no(\s|$)/i
      };
      const candidates = Array.from(root.querySelectorAll('button,[role="radio"],input[type="radio"],label,div,span'));
      const comboText = el => ((el.getAttribute?.('aria-label') || '') + ' ' + (el.textContent || '')).trim();
      let yes = candidates.find(e => tokenRx.yes.test(comboText(e)));
      let no  = candidates.find(e => tokenRx.no.test(comboText(e)));
      if (!yes || !no) {
        // Inputs by value or associated label
        const inputs = Array.from(root.querySelectorAll('input[type="radio"]'));
        for (const r of inputs) {
          const lbl = (r.id && (document.querySelector(`label[for="${r.id}"]`)?.textContent||'')) || (r.closest('label')?.textContent||'');
          if (!yes && (tokenRx.yes.test(r.value||'') || tokenRx.yes.test(lbl))) yes = r;
          if (!no  && (tokenRx.no.test(r.value||'')  || tokenRx.no.test(lbl)))  no  = r;
        }
      }
      return { yes, no };
    }

    const clickWithin = async (root, wantYes) => {
      try {
        const { yes, no } = findYesNoInRoot(root);
        let target = wantYes ? (yes || null) : (no || null);
        if (!target) return false;
        const clickable = target.closest && (target.closest('[role="radio"],button,label') || target) || target;
        try { clickable.scrollIntoView({ block: 'center' }); } catch(_) {}
        ['pointerdown','mousedown','mouseup','click'].forEach(t => clickable.dispatchEvent(new MouseEvent(t, { bubbles: true })));
        if (clickable.type === 'radio') { clickable.checked = true; clickable.dispatchEvent(new Event('input',{bubbles:true})); clickable.dispatchEvent(new Event('change',{bubbles:true})); }
        return true;
      } catch(_) { return false; }
    };

    for (let round = 0; round < maxRounds; round++) {
      const groups = getGroups().filter(g => !isAnswered(g));
      if (groups.length === 0) return true;
      for (const g of groups) {
        const want = decide(g.textContent||'');
        if (want === null) { await new Promise(r => setTimeout(r, groupDelay)); continue; }
        let ok = false; let attempts = 0;
        while (!ok && attempts < 6) {
          ok = await clickWithin(g, !!want);
          if (!ok) await new Promise(r => setTimeout(r, 250));
          attempts++;
        }
        await new Promise(r => setTimeout(r, groupDelay));
      }
      await new Promise(r => setTimeout(r, roundDelay));
    }
    return false;
  } catch(_) { return false; }
}

// Console event: persistent retry loop until all filled
window.addEventListener('JOB_AUTOFILL_ALL_YES_NO_PERSIST', () => { fillAllRadiosUntilComplete(12, 500, 1000); });

// One-shot: set "Do you identify as transgender?" to No
function setTransgenderNoOnce() {
  try {
    const transRx = /(do\s*you\s*identify\s*as\s*transgender)/i;
    return clickRadioByQuestion(transRx, false);
  } catch(_) { return false; }
}
window.addEventListener('JOB_AUTOFILL_TRANS_NO', () => {
  try { console.log('TRANS> set No:', setTransgenderNoOnce()); } catch(_) {}
});
